package ru.yandex.webmaster3.api.http.rest.routing;

import ru.yandex.autodoc.common.doc.types.ValueType;
import ru.yandex.webmaster3.api.http.common.request.converters.ApiQueryIdConverter;
import ru.yandex.webmaster3.api.http.common.request.converters.ApiSitemapIdConverter;
import ru.yandex.webmaster3.api.http.common.request.converters.ApiWebmasterHostIdConverter;
import ru.yandex.webmaster3.api.http.rest.types.WebmasterApiTypes;
import ru.yandex.webmaster3.api.sitemap.data.ApiSitemapId;
import ru.yandex.webmaster3.api.sitemap.util.SitemapIdUtil;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.ParameterConverter;
import ru.yandex.webmaster3.core.http.PrimitiveParameterConverter;
import ru.yandex.webmaster3.core.searchquery.QueryId;
import ru.yandex.webmaster3.core.util.functional.Bijection;

import java.lang.reflect.Type;
import java.util.UUID;
import java.util.function.Function;

/**
 * @author avhaliullin
 */
public class UrlParam<T> implements Bijection<String, T> {
    private final ParameterConverter parameterConverter;
    private final Class<T> parameterClass;
    private final Function<T, String> toUrlParamMapper;

    UrlParam(ParameterConverter parameterConverter, Class<T> parameterClass, Function<T, String> toUrlParamMapper) {
        this.parameterConverter = parameterConverter;
        this.parameterClass = parameterClass;
        this.toUrlParamMapper = toUrlParamMapper;
    }

    @Override
    public T leftToRight(String s) {
        return (T) parameterConverter.convert(s, parameterClass);
    }

    @Override
    public String rightToLeft(T t) {
        return toUrlParamMapper.apply(t);
    }

    public static final UrlParam<WebmasterHostId> HOST_ID = new UrlParam<>(new ApiWebmasterHostIdConverter(), WebmasterHostId.class, WebmasterHostId::toStringId);
    public static final UrlParam<String> STRING = new UrlParam<>(new StringParamConverter(), String.class, Function.identity());
    public static final UrlParam<Long> LONG = new UrlParam<>(new PrimitiveParameterConverter(), Long.class, Object::toString);
    public static final UrlParam<ApiSitemapId> SITEMAP_ID = new UrlParam<>(new ApiSitemapIdConverter(), ApiSitemapId.class, SitemapIdUtil::toString);
    public static final UrlParam<UUID> UUID_PARAM = new UrlParam<>(new UUIDParamConverter(), UUID.class, UUID::toString);
    public static final UrlParam<QueryId> QUERY_ID_PARAM = new UrlParam<>(new ApiQueryIdConverter(), QueryId.class, QueryId::toString);

    public ParameterConverter getParameterConverter() {
        return parameterConverter;
    }

    public Class<T> getParameterClass() {
        return parameterClass;
    }

    public static class StringParamConverter implements ParameterConverter {
        @Override
        public ValueType describeType(Type type) {
            return ValueType.STRING;
        }

        @Override
        public Object convert(String value, Type toType) {
            if (toType.equals(String.class)) {
                return value;
            }
            return null;
        }
    }

    public static class UUIDParamConverter implements ParameterConverter {
        @Override
        public ValueType describeType(Type type) {
            if (type == UUID.class) {
                return WebmasterApiTypes.UUID;
            }
            return null;
        }

        @Override
        public Object convert(String value, Type toType) {
            if (toType == UUID.class) {
                UUID result = UUID.fromString(value);
                return result;
            }
            return null;
        }
    }
}
