package ru.yandex.webmaster3.api.indexing.action;

import java.util.Collections;
import java.util.EnumMap;
import java.util.List;
import java.util.Map;
import java.util.function.ToLongFunction;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.common.data.ApiHistoryPoint;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.indexing.data.ApiIndexingIndicator;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.searchurl.history.dao.SiteStructureCHDao;
import ru.yandex.webmaster3.storage.searchurl.history.dao.SiteStructureCHDao.SiteStructureRecord;

/**
 * @author avhaliullin
 */
@Slf4j
@Description("Получить историю индексирования сайта")
@Category("indexing")
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@ActionPermission(Permission.COMMON)
public class IndexingHistoryAction extends AbstractApiAction<IndexingHistoryRequest, IndexingHistoryResponse> {

    private final SiteStructureCHDao siteStructureCHDao;

    @Override
    public IndexingHistoryResponse process(IndexingHistoryRequest request) {
        List<SiteStructureRecord> rawData = siteStructureCHDao.getSiteStructures(
                request.getHostId(),
                request.getDateFrom() == null ? null : request.getDateFrom().toInstant(),
                request.getDateTo() == null ? null : request.getDateTo().toInstant(),
                Collections.singleton(NewSiteStructure.ROOT_NODE_ID));
        Map<ApiIndexingIndicator, List<ApiHistoryPoint<Long>>> result = new EnumMap<>(ApiIndexingIndicator.class);
        for (ApiIndexingIndicator indicator : request.getIndexingIndicators()) {
            ToLongFunction<SiteStructureRecord> valueExtractor = indicatorExtractor(indicator);
            List<ApiHistoryPoint<Long>> indicatorHistory = rawData.stream()
                    .map(record -> {
                                long value = valueExtractor.applyAsLong(record);
                                return new ApiHistoryPoint<>(
                                        record.getTimestamp().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE),
                                        value
                                );
                            }
                    ).collect(Collectors.toList());
            result.put(indicator, indicatorHistory);
        }
        return new IndexingHistoryResponse(result);
    }

    private ToLongFunction<SiteStructureRecord> indicatorExtractor(ApiIndexingIndicator indicator) {
        if (indicator == ApiIndexingIndicator.SEARCHABLE) {
            return SiteStructureRecord::getNumOfDocsOnSearch;
        } else {
            return record -> record.getHttpCodes().entrySet().stream()
                    .filter(entry -> indicator.contains(entry.getKey()))
                    .mapToLong(Map.Entry::getValue)
                    .sum();
        }
    }

}
