package ru.yandex.webmaster3.api.indexing2.action;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.common.data.ApiHistoryPoint;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.indexing2.data.IndexingStatusEnum;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.indexing2.history.IndexedUrlsCountHistoryService;
import ru.yandex.webmaster3.storage.indexing2.history.data.IndexingHistoryData;

import java.util.*;
import java.util.function.BinaryOperator;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import static java.util.stream.Collectors.toMap;

/**
 * @author avhaliullin
 */
@Description("История обхода страниц сайта")
@Category("indexing")
@Component
public class Indexing2HistoryAction extends AbstractApiAction<Indexing2HistoryRequest, Indexing2HistoryResponse> {
    private IndexedUrlsCountHistoryService indexedUrlsCountHistoryService;

    public Indexing2HistoryAction() {
        super(Permission.COMMON);
    }

    @Override
    public Indexing2HistoryResponse process(Indexing2HistoryRequest request) {
        DateTime dateFrom = request.getDateFrom();
        DateTime dateTo = request.getDateTo();

        Map<Integer, IndexingHistoryData> rawHistory =
                indexedUrlsCountHistoryService.getIndexedUrlsCountHistory(
                        request.getHostId(),
                        NewSiteStructure.ROOT_NODE_ID,
                        dateFrom == null? null : dateFrom.toInstant(),
                        dateTo == null? null : dateTo.toInstant()
                );

        // Обьединяет две (Instant, count) мапы в одну, суммируя значения для count
        BinaryOperator<Map<Instant, Long>> mapMerger = (m1, m2) -> Stream.of(m1, m2)
                .map(Map::entrySet)
                .flatMap(Collection::stream)
                .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue, Long::sum, HashMap::new));

        // Переводит мапу в которой ключ это http код, в мапу в которой ключ - IndexingStatusEnum
        Map<IndexingStatusEnum, Map<Instant, Long>> indicatorsMap = rawHistory.entrySet().stream()
                .collect(toMap(
                        e -> IndexingStatusEnum.of(e.getKey()),
                        e -> e.getValue().getTimestamp2Value(),
                        mapMerger,
                        () -> new EnumMap<>(IndexingStatusEnum.class)
                ));

        Function<Map<Instant, Long>, List<ApiHistoryPoint<Long>>> toHistoryPointList = m -> m.entrySet().stream()
                .map(e -> new ApiHistoryPoint<>(e.getKey().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE), e.getValue()))
                .sorted(Comparator.comparing(ApiHistoryPoint::getDate))
                .collect(Collectors.toList());

        Map<IndexingStatusEnum, List<ApiHistoryPoint<Long>>> indicators = indicatorsMap.entrySet().stream()
                .collect(Collectors.toMap(
                        e -> e.getKey(),
                        e -> toHistoryPointList.apply(e.getValue())
                ));

        return new Indexing2HistoryResponse(indicators);
    }

    @Required
    public void setIndexedUrlsCountHistoryService(IndexedUrlsCountHistoryService indexedUrlsCountHistoryService) {
        this.indexedUrlsCountHistoryService = indexedUrlsCountHistoryService;
    }
}
