package ru.yandex.webmaster3.api.links.action;

import lombok.extern.slf4j.Slf4j;
import org.joda.time.LocalDate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.common.data.ApiHistoryPoint;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.links.data.ApiInternalLinksBrokenIndicator;
import ru.yandex.webmaster3.core.codes.ErrorGroupEnum;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.links.LinksService;

import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * @author leonidrom
 */

/*
 * Путь: /user/{user-id}/hosts/{host-id}/links/internal/broken/history/
 */
@Component
@Description("Получить историю количества неработающих внутренних ссылок для сайта")
@Category("links")
@Slf4j
public class InternalLinksBrokenHistoryAction extends AbstractApiAction<InternalLinksBrokenHistoryRequest, InternalLinksBrokenHistoryResponse> {
    private final LinksService linksService;

    @Autowired
    public InternalLinksBrokenHistoryAction(LinksService linksService) {
        super(Permission.EXTERNAL_LINKS);
        this.linksService = linksService;
    }

    @Override
    public InternalLinksBrokenHistoryResponse process(InternalLinksBrokenHistoryRequest request) {
        var hostId = request.getHostId();
        var dateFrom = request.getDateFromOrNow();
        var dateTo = request.getDateToOrNow();
        Map<HttpCodeInfo, NavigableMap<LocalDate, Long>> httpCodeHistory = linksService.getInternalHttpHistory(
                hostId, dateFrom, dateTo, true, LinksService.INTERNAL_LINKS_BROKEN_HISTORY_PREDICATE);

        Map<ApiInternalLinksBrokenIndicator, NavigableMap<LocalDate, Long>> result = new EnumMap<>(
                ApiInternalLinksBrokenIndicator.class);
        httpCodeHistory.forEach((key, value) -> {
            // наличие значения в ErrorGroupEnum гарантируется предикатом INTERNAL_LINKS_BROKEN_HISTORY_PREDICATE
            var apiIndicator = ApiInternalLinksBrokenIndicator.fromErrorCodeGroup(ErrorGroupEnum.get(key.getCode()).get());
            NavigableMap<LocalDate, Long> groupHistory = result.computeIfAbsent(apiIndicator, g -> new TreeMap<>());

            // смержим две мапы
            groupHistory = Stream.of(groupHistory, value)
                    .flatMap(m -> m.entrySet().stream())
                    .collect(Collectors.toMap(
                            Map.Entry::getKey, Map.Entry::getValue,
                            Long::sum, TreeMap::new));

            // и положим обратно
            result.put(apiIndicator, groupHistory);
        });

        Map<ApiInternalLinksBrokenIndicator, List<ApiHistoryPoint<Long>>> indicatorsHistory = new HashMap<>();
        result.forEach((apiIndicator, historyMap) -> {
            var indicatorHistoryPoints = historyMap.entrySet().stream()
                    .filter(entry -> entry.getValue() != 0) // оставим только те даты, где были какие то битые ссылки
                    .map(entry -> new ApiHistoryPoint<>(entry.getKey().toDateTimeAtStartOfDay(), entry.getValue()))
                    .collect(Collectors.toList());

            if (!indicatorHistoryPoints.isEmpty()) {
                indicatorsHistory.put(apiIndicator, indicatorHistoryPoints);
            }
        });

        return new InternalLinksBrokenHistoryResponse(indicatorsHistory);
    }
}
