package ru.yandex.webmaster3.api.searchurls.action;

import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.NavigableMap;
import java.util.Set;

import org.joda.time.DateTime;
import org.joda.time.Instant;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.common.data.ApiHistoryPoint;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.searchurls.data.ApiSearchEventEnum;
import ru.yandex.webmaster3.core.http.UserContext;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitestructure.NewSiteStructure;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.searchurl.history.SearchUrlHistoryService;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlHistoryIndicator;
import ru.yandex.webmaster3.storage.searchurl.history.data.SearchUrlHistoryPoint;

/**
 * @author avhaliullin
 */
@Description("История появления/удаления страниц сайта на поиске")
@Category("searchurls")
@Component
public class SearchUrlEventsHistoryAction extends AbstractApiAction<SearchUrlEventsHistoryRequest, SearchUrlEventsHistoryResponse> {
    private static Set<SearchUrlHistoryIndicator> EVENT_INDICATORS = EnumSet.of(SearchUrlHistoryIndicator.GONE, SearchUrlHistoryIndicator.NEW);

    private SearchUrlHistoryService searchUrlHistoryService;

    public SearchUrlEventsHistoryAction() {
        super(Permission.COMMON);
    }

    @Override
    public SearchUrlEventsHistoryResponse process(SearchUrlEventsHistoryRequest request) {
        DateTime dateFrom = request.getDateFrom();
        DateTime dateTo = request.getDateTo();
        UserContext.setUserId(request.getUserId());
        // получим историю для ROOT_NODE_ID
        NavigableMap<Instant, SearchUrlHistoryPoint> rawHistory = searchUrlHistoryService.getSearchHistory(
                request.getHostId(),
                Collections.singleton(NewSiteStructure.ROOT_NODE_ID),
                EVENT_INDICATORS,
                dateFrom == null? null : dateFrom.toInstant(),
                dateTo == null? null : dateTo.toInstant()
        );

        // и приведем ее к нужному виду
        Map<ApiSearchEventEnum, List<ApiHistoryPoint<Long>>> apiIndicatorsMap = new HashMap<>();
        for (Map.Entry<Instant, SearchUrlHistoryPoint> entry : rawHistory.entrySet()) {
            DateTime historyDate = entry.getKey().toDateTime(TimeUtils.EUROPE_MOSCOW_ZONE);
            Map<Long, Map<SearchUrlHistoryIndicator, Long>> node2Values = entry.getValue().getNode2Values();

            // нас интересует только история для ROOT_NODE_ID
            Map<SearchUrlHistoryIndicator, Long> indicatorsMap = node2Values.get(NewSiteStructure.ROOT_NODE_ID);
            if (indicatorsMap == null) {
                continue;
            }

            // пройдемся по всем индикатором для данной даты
            for (Map.Entry<SearchUrlHistoryIndicator, Long> indicatorEntry : indicatorsMap.entrySet()) {
                // и добавим значение индикатора к списку
                ApiSearchEventEnum event = ApiSearchEventEnum.fromSearchUrlHistoryIndicator(indicatorEntry.getKey());
                List<ApiHistoryPoint<Long>> historyPoints = apiIndicatorsMap.computeIfAbsent(event, ign -> new ArrayList<>());
                historyPoints.add(new ApiHistoryPoint<>(historyDate, indicatorEntry.getValue()));
            }
        }

        return new SearchUrlEventsHistoryResponse(apiIndicatorsMap);
    }

    @Required
    public void setSearchUrlHistoryService(SearchUrlHistoryService searchUrlHistoryService) {
        this.searchUrlHistoryService = searchUrlHistoryService;
    }
}
