package ru.yandex.webmaster3.api.searchurls.data;

import org.jetbrains.annotations.Nullable;
import org.joda.time.DateTime;
import org.joda.time.LocalDate;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.searchurl.samples.data.SearchUrlEventSample;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.Optional;

/**
 * @author avhaliullin
 */
public class ApiSearchEventSample {
    private final URL url;
    private final String title;
    private final DateTime eventDate;
    private final DateTime lastAccess;
    private final ApiSearchEventEnum event;

    @Nullable
    private final ApiExcludedSearchUrlInfo excludedInfo;

    private ApiSearchEventSample(URL url, String title, DateTime eventDate, DateTime lastAccess, ApiSearchEventEnum event,
                                 ApiExcludedSearchUrlInfo excludedInfo) {
        this.url = url;
        this.title = title;
        this.eventDate = eventDate;
        this.lastAccess = lastAccess;
        this.event = event;
        this.excludedInfo = excludedInfo;
    }

    @Description("адрес страницы")
    public URL getUrl() {
        return url;
    }

    @Description("заголовок страницы")
    public String getTitle() {
        return title;
    }

    @Description("дата события (добавления или исключения с поиска)")
    public DateTime getEventDate() {
        return eventDate;
    }

    @Description("дата последнего обхода страницы на момент события")
    public DateTime getLastAccess() {
        return lastAccess;
    }

    @Description("событие")
    public ApiSearchEventEnum getEvent() {
        return event;
    }

    @Description("причина исключения страницы с поиска")
    public Optional<ApiExcludedUrlStatusEnum> getExcludedUrlStatus() {
        return Optional.ofNullable(excludedInfo).map(ApiExcludedSearchUrlInfo::getExcludedUrlStatus);
    }

    @Description("HTTP-код овета страницы для статуса HTTP_ERROR")
    public Optional<Integer> getBadHttpStatus() {
        return Optional.ofNullable(excludedInfo).map(ApiExcludedSearchUrlInfo::getBadHttpStatus);
    }

    @Description("другой адрес, по которому страница известна роботу - цель редиректа, значение мета-тега canonical или основной контентный дубль")
    public Optional<URL> getTargetUrl() {
        return Optional.ofNullable(excludedInfo).map(ApiExcludedSearchUrlInfo::getTargetUrl);
    }

    @Nullable
    public static ApiSearchEventSample fromSearchUrlEventSample(SearchUrlEventSample sample) {
        URL url;
        try {
            url = new URL(sample.getUrl());
        } catch (MalformedURLException e) {
            // такого быть не должно
            return null;
        }

        var event = ApiSearchEventEnum.fromSearchUrlEventType(sample.getEventType());
        var excludedInfo = ApiExcludedSearchUrlInfo.fromUrlStatusInfo(sample.getStatusInfo());

        final LocalDate searchBaseDate = sample.getSearchBaseDate();
        final LocalDate lastAccess = sample.getLastAccess();
        return new ApiSearchEventSample(
                url, sample.getTitle(),
                searchBaseDate.toDateTimeAtStartOfDay(TimeUtils.EUROPE_MOSCOW_ZONE),
                lastAccess != null ? lastAccess.toDateTimeAtStartOfDay(TimeUtils.EUROPE_MOSCOW_ZONE) : null,
                event, excludedInfo
        );
    }
}
