package ru.yandex.webmaster3.api.sitemap.action;

import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.sitemap.data.ApiSitemapId;
import ru.yandex.webmaster3.api.sitemap.data.ApiSitemapInfo;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.core.sitemap.HostSitemap;
import ru.yandex.webmaster3.storage.sitemap.dao.SitemapsCHDao;

/**
 * @author avhaliullin
 */
@Description("Получить список файлов sitemap, обнаруженных роботами Яндекса")
@Category("sitemap")
@Component
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@ActionPermission(Permission.COMMON)
public class GetSitemapListAction extends AbstractApiAction<GetSitemapListRequest, GetSitemapListResponse> {

    private final SitemapsCHDao sitemapsCHDao;

    @Override
    public GetSitemapListResponse process(GetSitemapListRequest request) {
        Optional<ApiSitemapId> parentIdOpt = request.getLocator().getParentId();
        UUID parentId = parentIdOpt.map(ApiSitemapId::getId).orElse(null);
        UUID from = null;
        if (request.getFrom() != null) {
            if (!Objects.equals(request.getFrom().getParentId(), parentId)) {
                // не сходится parent, по которому мы просматриваем, и parent из from id - можно даже не ходить в базу
                return new GetSitemapListResponse(Collections.emptyList());
            }
            from = request.getFrom() == null ? null : request.getFrom().getId();
        }
        List<HostSitemap> sitemaps = sitemapsCHDao.getSitemaps(request.getHostId(), parentId, from, request.getLimit());

        return new GetSitemapListResponse(
                sitemaps.stream()
                        .map(info -> ApiSitemapInfo.fromHostSitemap(parentId, info))
                        .collect(Collectors.toList())
        );
    }
}
