package ru.yandex.webmaster3.api.turbo.action;

import org.joda.time.DateTime;
import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.rest.response.ApiPostResponse;
import ru.yandex.webmaster3.api.http.rest.response.ApiResponse;
import ru.yandex.webmaster3.api.http.rest.response.errors.ApiErrorCode;
import ru.yandex.webmaster3.api.http.rest.response.errors.CommonApiErrors;
import ru.yandex.webmaster3.api.http.rest.response.meta.ResponseWithLocation;
import ru.yandex.webmaster3.api.turbo.GetTurboPushTaskStatusLocator;
import ru.yandex.webmaster3.core.data.WebmasterHostId;

import java.util.UUID;

/**
 * Created by Oleg Bazdyrev on 22/08/2017.
 */
public interface AddTurboPagesResponse extends ApiPostResponse {
    class NormalResponse extends ApiPostResponse.Accepted202 implements AddTurboPagesResponse, ResponseWithLocation<GetTurboPushTaskStatusLocator> {
        private final long userId;
        private final WebmasterHostId hostId;
        private final UUID taskId;

        public NormalResponse(long userId, WebmasterHostId hostId, UUID taskId) {
            this.userId = userId;
            this.hostId = hostId;
            this.taskId = taskId;
        }

        @Description("идентификатор задачи")
        public String getTaskId() {
            return taskId.toString();
        }

        @Override
        public GetTurboPushTaskStatusLocator getLocation() {
            return new GetTurboPushTaskStatusLocator(userId, hostId, taskId);
        }
    }

    enum Errors implements ApiErrorCode {
        TOO_MANY_REQUESTS_ERROR,
        LICENCE_NOT_ACCEPTED,
        REQUEST_ENTITY_TOO_LARGE,
        UPLOAD_ADDRESS_EXPIRED,
    }

    @Description("Превышен лимит запросов")
    class TooManyRequestsError extends ApiResponse.TooManyRequests429<Errors> implements AddTurboPagesResponse {
        public TooManyRequestsError(String message) {
            super(Errors.TOO_MANY_REQUESTS_ERROR, message);
        }
    }

    @Description("Не принята оферта")
    class LicenceNotAcceptedError extends ApiResponse.Forbidden403<Errors> implements AddTurboPagesResponse {
        public LicenceNotAcceptedError() {
            super(Errors.LICENCE_NOT_ACCEPTED, "Licence not accepted");
        }
    }

    @Description("Передан слишком большой RSS-фид")
    class RequestEntityTooLarge extends ApiResponse.RequestEntityTooLarge413<Errors> implements AddTurboPagesResponse {
        public RequestEntityTooLarge(String message) {
            super(Errors.REQUEST_ENTITY_TOO_LARGE, message);
        }
    }

    @Description("Тело запроса не прошло валидацию")
    class EntityValidationError extends BadRequest400<CommonApiErrors.Code> implements AddTurboPagesResponse {
        public EntityValidationError(String message) {
            super(CommonApiErrors.Code.ENTITY_VALIDATION_ERROR, message);
        }
    }

    class AddressExpiredError extends ApiResponse.Gone410<Errors> implements AddTurboPagesResponse {
        private final DateTime validUntil;
        public AddressExpiredError(DateTime validUntil) {
            super(Errors.UPLOAD_ADDRESS_EXPIRED, "Upload address was valid until" + validUntil);
            this.validUntil = validUntil;
        }

        @Description("Дата, до которой адес был валиден")
        public DateTime getValidUntil() {
            return validUntil;
        }
    }
}
