package ru.yandex.webmaster3.api.verification.action;

import java.util.Optional;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.api.http.auth.ActionPermission;
import ru.yandex.webmaster3.api.http.auth.Permission;
import ru.yandex.webmaster3.api.http.rest.AbstractApiAction;
import ru.yandex.webmaster3.api.verification.data.ApiExplicitVerificationType;
import ru.yandex.webmaster3.api.verification.data.ApiVerificationState;
import ru.yandex.webmaster3.api.verification.data.ApiVerificationType;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.host.verification.UinUtil;
import ru.yandex.webmaster3.core.host.verification.UserHostVerificationInfo;
import ru.yandex.webmaster3.core.host.verification.VerificationCausedBy;
import ru.yandex.webmaster3.core.host.verification.VerificationType;
import ru.yandex.webmaster3.core.host.verification.VerificationTypeScope;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.Category;
import ru.yandex.webmaster3.storage.user.dao.UserHostVerificationYDao;
import ru.yandex.webmaster3.storage.verification.HostVerifierService;

/**
 * @author avhaliullin
 */
@Description("Запустить процедуру подтверждения прав")
@Category("verification")
@Component
@ActionPermission(Permission.COMMON)
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class VerifyHostAction extends AbstractApiAction<VerifyHostRequest, VerifyHostResponse> {

    private final UserHostVerificationYDao userHostVerificationYDao;
    private final HostVerifierService hostVerifierService;

    @Override
    public VerifyHostResponse process(VerifyHostRequest request) {
        VerificationType verificationType = request.getVerificationType().getVerificationType();
        Preconditions.checkArgument(verificationType.visibleIn(VerificationTypeScope.API));
        UserHostVerificationInfo verificationInfo = userHostVerificationYDao.getLatestRecord(request.getUserId(), request.getHostId());

        if (verificationInfo == null) {
            return new VerifyHostResponse.NoVerificationRecord();
        }

        if (HostVerifierService.isVerificationInProgress(verificationInfo)) {
            return new VerifyHostResponse.VerificationAlreadyInProgress(
                    ApiExplicitVerificationType.fromVerificationTypeOrNull(verificationInfo.getVerificationType())
            );
        }

        boolean launched = hostVerifierService.initiateVerification(
                verificationInfo,
                VerificationCausedBy.INITIAL_VERIFICATION,
                request.getHostId(),
                request.getUserId(),
                verificationType
        );
        if (!launched) {
            // Сейчас (июль 2018) launched будет равен true для всех разрешённых в API типов подтверждений,
            // но на всякий случай бросим исключение
            throw new WebmasterException("Failed to trigger verification",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null));
        }
        Set<ApiExplicitVerificationType> isApplicable = ApiExplicitVerificationType.fromVerificationTypes(
                hostVerifierService.listApplicable(request.getHostId(), request.getUserId(), VerificationTypeScope.API)
        );
        return new VerifyHostResponse.NormalResponse(
                UinUtil.getUinString(verificationInfo.getVerificationUin()),
                ApiVerificationState.IN_PROGRESS,
                Optional.of(ApiVerificationType.fromVerificationType(verificationType)),
                Optional.empty(),
                Optional.empty(),
                isApplicable
        );
    }
}
