package ru.yandex.webmaster3.coordinator;

import org.joda.time.DateTime;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

/**
 * @author aherman
 */
public class ReceiverStatusService {
    private final Map<ReceiverType, MutableReceiverStatus> receiverStatus;

    public ReceiverStatusService() {
        Map<ReceiverType, MutableReceiverStatus> result = new HashMap<>();
        for (ReceiverType receiverType : ReceiverType.values()) {
            result.putIfAbsent(receiverType, new MutableReceiverStatus());
        }

        receiverStatus = Collections.unmodifiableMap(result);
    }

    public boolean isReceiverEnabled(ReceiverType receiverType) {
        DateTime disableDate = receiverStatus.get(receiverType).disableDate.get();
        return disableDate == null;
    }

    public void changeReceiverStatus(ReceiverType receiverType, boolean enable) {
        if (enable) {
            receiverStatus.get(receiverType).disableDate.set(null);
        } else {
            receiverStatus.get(receiverType).disableDate.set(DateTime.now());
        }
    }

    public void messageSkipped(ReceiverType receiverType) {
        receiverStatus.get(receiverType).messageSkipped.incrementAndGet();
    }

    public void messageReceived(ReceiverType receiverType) {
        receiverStatus.get(receiverType).messageReceived.incrementAndGet();
    }

    public void messageProcessed(ReceiverType receiverType) {
        receiverStatus.get(receiverType).messageProcessed.incrementAndGet();
    }

    public Map<ReceiverType, ReceiverStatus> getStatus() {
        HashMap<ReceiverType, ReceiverStatus> result = new HashMap<>();
        for (ReceiverType receiverType : ReceiverType.values()) {
            result.put(receiverType, receiverStatus.get(receiverType).toStatus());
        }
        return result;
    }

    public static enum ReceiverType {
        SITE_STRUCTURE,
        SITEMAP,
        HOST_SITEMAP_RELATION,
        QUERY_STATISTICS,
        LINKS_STATISTICS,
    }

    private static class MutableReceiverStatus {
        private final AtomicReference<DateTime> disableDate = new AtomicReference<>();
        private final AtomicLong messageSkipped = new AtomicLong();
        private final AtomicLong messageReceived = new AtomicLong();
        private final AtomicLong messageProcessed = new AtomicLong();

        public ReceiverStatus toStatus() {
            return new ReceiverStatus(
                    disableDate.get(),
                    messageSkipped.get(),
                    messageReceived.get(),
                    messageProcessed.get()
            );
        }
    }

    public static class ReceiverStatus {
        private final DateTime disableDate;
        private final long messageSkipped;
        private final long messageReceived;
        private final long messageProcessed;

        public ReceiverStatus(DateTime disableDate, long messageSkipped, long messageReceived, long messageProcessed) {
            this.disableDate = disableDate;
            this.messageSkipped = messageSkipped;
            this.messageReceived = messageReceived;
            this.messageProcessed = messageProcessed;
        }

        public DateTime getDisableDate() {
            return disableDate;
        }

        public long getMessageSkipped() {
            return messageSkipped;
        }

        public long getMessageReceived() {
            return messageReceived;
        }

        public long getMessageProcessed() {
            return messageProcessed;
        }
    }
}
