package ru.yandex.webmaster3.coordinator.http;

import java.util.HashSet;
import java.util.Set;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableLong;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ActionStatus;
import ru.yandex.webmaster3.storage.host.dao.HostsYDao;
import ru.yandex.webmaster3.storage.user.service.UserHostsService;

/**
 * @author avhaliullin
 */
@Slf4j
@Component("/tmp/removeUnverified")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RemoveUnverifiedAction extends Action<RemoveUnverifiedAction.Req, RemoveUnverifiedAction.Res> {

    private final HostsYDao hostsYDao;
    private final UserHostsService userHostsService;

    @Override
    public Res process(Req request) throws WebmasterException {
        try {
            Set<WebmasterHostId> allHosts = new HashSet<>();
            hostsYDao.forEach(pair -> {
                allHosts.add(pair.getLeft());
                if (allHosts.size() % 10000 == 0) {
                    log.info("Loaded {} hosts", allHosts.size());
                }
            });

            MutableLong vh = new MutableLong();
            userHostsService.forEach(uh -> {
                if (vh.incrementAndGet() % 10000 == 0) {
                    log.info("Loaded {} u-h pairs, {} hosts in set", vh.getValue(), allHosts.size());
                }
                allHosts.remove(uh.getValue().getWebmasterHostId());
            });
            log.info("{} hosts to remove", allHosts.size());
            Thread.sleep(60000);
            for (WebmasterHostId hostId : allHosts) {
                try {
                    log.info("Deleting host {}", hostId);
                    hostsYDao.deleteHost(hostId);
                } catch (Exception e) {
                    log.error("Failed to check host", e);
                }
            }
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
        return new Res();
    }

    public static class Req implements ActionRequest {}

    public static class Res implements ActionResponse {
        @Override
        public ActionStatus getRequestStatus() {
            return ActionStatus.SUCCESS;
        }
    }
}
