package ru.yandex.webmaster3.coordinator.http.logging;

import java.io.IOException;
import java.util.UUID;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.joda.time.DateTime;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.HostNameAwareActionRequest;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.storage.logging.TaskLogEntry;
import ru.yandex.webmaster3.storage.logging.TasksLoggingService;

/**
 * @author tsyplyaev
 */
public class AddPeriodicTaskLogEntry extends Action<AddPeriodicTaskLogEntry.Request, AddPeriodicTaskLogEntry.Response> {
    private static final Logger log = LoggerFactory.getLogger(AddPeriodicTaskLogEntry.class);

    private TasksLoggingService periodicTasksLoggingService;
    private static final ObjectMapper OM = new ObjectMapper();

    @Override
    public Response process(Request request) throws WebmasterException {
        try {
            JsonNode state = null;
            try {
                String c = request.getState();
                if (!c.isEmpty()) {
                    state = OM.readTree(c);
                }
            } catch (IOException ex) {
                log.error("JSON error while parsing 'state', runId = {}", request.getRunId(), ex);
            }

            JsonNode error = null;
            try {
                String c = request.getError();
                if (!c.isEmpty()) {
                    error = OM.readTree(c);
                }
            } catch (IOException ex) {
                log.error("JSON error while parsing 'error', runId = {}", request.getRunId(), ex);
            }

            periodicTasksLoggingService.addEntry(request.getDate() != null ? request.getDate() : DateTime.now(),
                    request.getTaskType(), request.getEventType(), request.getHostName(), request.getRunId(), request.getRunTime(), state, error, null);

            return new Response();
        } catch (Exception ex) {
            throw new WebmasterException("Error running task",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), ex.getMessage()), ex);
        }
    }

    @Required
    public void setPeriodicTasksLoggingService(TasksLoggingService periodicTasksLoggingService) {
        this.periodicTasksLoggingService = periodicTasksLoggingService;
    }

    public static class Request implements HostNameAwareActionRequest {
        private DateTime date;
        private TaskLogEntry.EventType eventType;
        private String taskType;
        private String hostName;
        private UUID runId;
        private Long runTime;
        private String state;
        private String error;

        public String getHostName() {
            return hostName;
        }

        @Override
        public void setHostName(String hostName) {
            this.hostName = hostName;
        }

        @RequestQueryProperty
        public void setDate(DateTime date) {
            this.date = date;
        }

        public DateTime getDate() {
            return date;
        }

        public TaskLogEntry.EventType getEventType() {
            return eventType;
        }

        @RequestQueryProperty(required = true)
        public void setEventType(TaskLogEntry.EventType eventType) {
            this.eventType = eventType;
        }

        public String getTaskType() {
            return taskType;
        }

        @RequestQueryProperty(required = true)
        public void setTaskType(String taskType) {
            this.taskType = taskType;
        }

        public UUID getRunId() {
            return runId;
        }

        @RequestQueryProperty(required = true)
        public void setRunId(UUID runId) {
            this.runId = runId;
        }

        public Long getRunTime() {
            return runTime;
        }

        @RequestQueryProperty
        public void setRunTime(Long runTime) {
            this.runTime = runTime;
        }

        public String getState() {
            return state;
        }

        @RequestQueryProperty
        public void setState(String state) {
            this.state = state;
        }

        public String getError() {
            return error;
        }

        @RequestQueryProperty
        public void setError(String error) {
            this.error = error;
        }
    }

    public static class Response implements ActionResponse.NormalResponse {
    }
}
