package ru.yandex.webmaster3.coordinator.http.metrika;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.google.common.collect.Range;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Component;
import ru.yandex.webmaster3.coordinator.http.SimpleRequest;
import ru.yandex.webmaster3.coordinator.http.SimpleResponse;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrika.counters.AnonymousCounterBinding;
import ru.yandex.webmaster3.core.metrika.counters.CounterBinding;
import ru.yandex.webmaster3.core.metrika.counters.CounterBindingStateEnum;
import ru.yandex.webmaster3.core.metrika.counters.MetrikaCountersUtil;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.storage.metrika.MetrikaCounterBindingService;
import ru.yandex.webmaster3.storage.metrika.dao.MetrikaCounterBindingStateYDao;
import ru.yandex.webmaster3.storage.util.yt.*;

import java.util.HashMap;
import java.util.Map;

/**
 * @author leonidrom
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/tmp/metrika/fixMetrikaBindings")
public class FixMetrikaBindings extends Action<SimpleRequest, SimpleResponse> {
    private final YtService ytService;
    private final MetrikaCounterBindingStateYDao metrikaCounterBindingStateYDao;
    private final MetrikaCounterBindingService metrikaCounterBindingService;

    @Value("${external.yt.service.hahn}://home/metrika/webmaster/domains_counter_links")
    private final YtPath metrikaTablePath;

    @Override
    public SimpleResponse process(SimpleRequest request) throws WebmasterException {
        log.info("Started reading metrika table");
        Map<Pair<Long, String>, YtRow> metrikaTableBindingStates = new HashMap<>();
        ytService.inTransaction(metrikaTablePath)
                .withLock(metrikaTablePath, YtLockMode.SNAPSHOT)
                .execute(cypressService -> {
                    try {
                        readMetrikaTable(cypressService, metrikaTableBindingStates);
                    } catch (Exception e) {
                        throw new WebmasterException("Error reading Yt table",
                                new WebmasterErrorResponse.YTServiceErrorResponse(getClass(), e), e);
                    }

                    return true;
        });

        log.info("Finished reading metrika table: {}", metrikaTableBindingStates.size());

        log.info("Started fixing bindings");
        int diffCount = fixBindings(metrikaTableBindingStates);
        log.info("Finished fixing bindings: {}", diffCount);

        return new SimpleResponse();
    }

    private int fixBindings(Map<Pair<Long, String>, YtRow> metrikaTableBindingStates) {
        DateTime fixThreshold = DateTime.now().minusDays(14);
        var diffCount = new MutableInt();
        metrikaCounterBindingStateYDao.forEachLink(cb -> {
            if (cb.getUpdateDate().isAfter(fixThreshold)) {
                return;
            }

            var key = Pair.of(cb.getCounterId(), cb.getDomain());
            YtRow metrikaTableBinding = metrikaTableBindingStates.get(key);
            if (metrikaTableBinding == null) {
                diffCount.increment();
                log.info("No metrika binding for {}", key);
                return;
            }

            String wmState = cb.getCounterBindingState().getMeaning();
            String metrikaTableState = metrikaTableBinding.status;

            if (!wmState.equals(metrikaTableState)) {
                diffCount.increment();
                fixBinding(cb, metrikaTableBinding);
            }

        }, false);

        return diffCount.getValue();
    }

    private void fixBinding(CounterBinding cb, YtRow metrikaTableState) {
        if (metrikaTableState.status.equals("deleted") && cb.getCounterBindingState() != CounterBindingStateEnum.APPROVED) {
            log.info("Will fix binding {}, {}, {} vs {}", cb.getCounterId(), cb.getDomain(),
                    cb.getCounterBindingState().getMeaning(), metrikaTableState.status);


            metrikaCounterBindingService.updateStateWithMetrikaUser(
                    cb.getDomain(),
                    CounterBindingStateEnum.NONE,
                    cb.getCounterId(),
                    cb.getLastUserLogin(),
                    DateTime.now(TimeUtils.EUROPE_MOSCOW_ZONE),
                    metrikaTableState.metrikaUID,
                    AnonymousCounterBinding.ORIGIN_METRIKA);
        } else if (metrikaTableState.status.equals("ok") && cb.getCounterBindingState() == CounterBindingStateEnum.WEBMASTER_REQUEST) {
            log.info("Will fix binding {}, {}, {} vs {}", cb.getCounterId(), cb.getDomain(),
                    cb.getCounterBindingState().getMeaning(), metrikaTableState.status);


            metrikaCounterBindingService.updateStateWithMetrikaUser(
                    cb.getDomain(),
                    CounterBindingStateEnum.APPROVED,
                    cb.getCounterId(),
                    cb.getLastUserLogin(),
                    DateTime.now(TimeUtils.EUROPE_MOSCOW_ZONE),
                    metrikaTableState.metrikaUID,
                    AnonymousCounterBinding.ORIGIN_METRIKA);
        }
    }

    private void readMetrikaTable(YtCypressService cypressService, Map<Pair<Long, String>, YtRow> metrikaBindingStates)
            throws Exception {
        var reader = new AsyncTableReader<>(cypressService, metrikaTablePath, Range.all(),
                YtTableReadDriver.createYSONDriver(YtRow.class)).withRetry(3);

        try (var iterator = reader.read()) {
            while (iterator.hasNext()) {
                YtRow row = iterator.next();
                metrikaBindingStates.put(Pair.of(row.counterId, MetrikaCountersUtil.domainToCanonicalAscii(row.domain)), row);
            }
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    public static final class YtRow {
        private final long counterId;
        private final String domain;
        private final String status;
        private final String updateTime;
        private final long metrikaUID;
        private final long webmasterUID;

        @JsonCreator
        public YtRow(
                @JsonProperty("counter_id") long counterId,
                @JsonProperty("domain") String domain,
                @JsonProperty("status") String status,
                @JsonProperty("update_time") String updateTime,
                @JsonProperty("metrika_uid") long metrikaUID,
                @JsonProperty("webmaster_uid") long webmasterUID) {
            this.counterId = counterId;
            this.domain = domain;
            this.status = status;
            this.updateTime = updateTime;
            this.metrikaUID = metrikaUID;
            this.webmasterUID = webmasterUID;
        }
    }
}
