package ru.yandex.webmaster3.coordinator.http.migration;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.TreeMap;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.tuple.Pair;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.coordinator.http.SimpleRequest;
import ru.yandex.webmaster3.coordinator.http.SimpleResponse;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.util.W3Collectors;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemReplicasCHDao;
import ru.yandex.webmaster3.storage.clickhouse.system.dao.ClickhouseSystemTablesCHDao;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseHost;
import ru.yandex.webmaster3.storage.util.clickhouse2.ClickhouseQueryContext;
import ru.yandex.webmaster3.storage.util.clickhouse2.MdbClickhouseServer;

/**
 * Created by Oleg Bazdyrev on 13/04/2022.
 */
@Slf4j
@RequiredArgsConstructor(onConstructor_ = @Autowired)
@Component("/tmp/migrate/clickhouseVla")
public class MigrateClickhouseTablesToVlaTmpAction extends Action<SimpleRequest, SimpleResponse> {

    private static final int MAX_REPLICATION_PER_HOST = 3;

    private static final Set<String> DONOR_DCS = Set.of("SAS", "VLA");
    private static final String RECIPIENT_DC = "VLA-TMP";

    private final MdbClickhouseServer mdbClickhouseServer;
    private final ClickhouseSystemTablesCHDao clickhouseSystemTablesCHDao;
    private final ClickhouseSystemReplicasCHDao legacyMdbClickhouseSystemReplicasCHDao;

    @Override
    public SimpleResponse process(SimpleRequest request) throws WebmasterException {
        int shardCount = mdbClickhouseServer.getShardsCount();
        Map<Integer, TreeMap<String, String>> sourceTablesPerShard = new HashMap<>();

        for (int shard = 0; shard < shardCount; shard++) {
            int finalShard = shard;
            List<TreeMap<String, String>> tablesCreateScripts = new ArrayList<>();
            log.info("Processing shard {}", shard);
            for (String donorDc : DONOR_DCS) {
                ClickhouseHost host = mdbClickhouseServer.getHosts().stream().filter(
                        ch -> ch.getShard() == finalShard && ch.getDcName().equals(donorDc)).findAny().orElseThrow();

                tablesCreateScripts.add(
                        mdbClickhouseServer.queryAll(ClickhouseQueryContext.useDefaults().setHost(host),
                                "SELECT database || '.' || name, create_table_query FROM system.tables WHERE database LIKE 'webmaster%'",
                                chRow -> Pair.of(chRow.getString(0), chRow.getString(1))
                        ).stream().collect(W3Collectors.toTreeMap())
                );
            }
            TreeMap<String, String> tables = tablesCreateScripts.get(0);
            tables.keySet().retainAll(tablesCreateScripts.get(1).keySet());
            sourceTablesPerShard.put(shard, tables);
        }

        boolean hasTablesToReplicate;
        do {
            try {
                hasTablesToReplicate = false;
                for (int shard = 0; shard < shardCount; shard++) {
                    int finalShard = shard;
                    ClickhouseHost recipientHost = mdbClickhouseServer.getHosts().stream().filter(
                            ch -> ch.getShard() == finalShard && ch.getDcName().equals(RECIPIENT_DC)).findAny().orElseThrow();
                    TreeMap<String, String> tables = sourceTablesPerShard.get(shard);
                    log.info("Processing shard {}, tables remaining - {}", shard, tables.size());
                    Map<String, Integer> insertsByDbTable = legacyMdbClickhouseSystemReplicasCHDao.getInsertsInQueue(recipientHost, tables.keySet());
                    int replicationCount = 0;
                    for (Iterator<Map.Entry<String, String>> iterator = tables.entrySet().iterator(); iterator.hasNext();) {
                        Map.Entry<String, String> entry = iterator.next();
                        String name = entry.getKey();
                        String createScript = entry.getValue();
                        log.info("Processing table {}", name);
                        if (!createScript.contains("ENGINE = Replicated")) {
                            // just recreate
                            log.info("Recreating non-replicated table {}", name);
                            mdbClickhouseServer.execute(ClickhouseQueryContext.useDefaults().setHost(recipientHost),
                                    "DROP TABLE IF EXISTS " + name + " NO DELAY");
                            mdbClickhouseServer.execute(ClickhouseQueryContext.useDefaults().setHost(recipientHost), createScript
                                    .replace("CREATE TABLE", "CREATE TABLE IF NOT EXISTS")
                                    .replace("CREATE VIEW", "CREATE VIEW IF NOT EXISTS"));
                            iterator.remove();
                        } else {
                            Integer inserts = insertsByDbTable.get(name);
                            if (inserts == null) {
                                log.info("Creating table {}", name);
                                mdbClickhouseServer.execute(ClickhouseQueryContext.useDefaults().setHost(recipientHost), createScript);
                                replicationCount++;
                                hasTablesToReplicate = true;
                            } else if (inserts > 0) {
                                log.info("Table");
                                replicationCount++;
                                hasTablesToReplicate = true;
                            } else {
                                log.info("Table {} already created and replicated", name);
                                iterator.remove();
                            }
                        }
                        if (replicationCount >= MAX_REPLICATION_PER_HOST) {
                            log.info("Max replications count on shard {} reached", shard);
                            break;
                        }
                    }
                }
            } catch (Exception e) {
                log.error("Error: ", e);
                hasTablesToReplicate = true;

            }
            try {
                Thread.sleep(30000);
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        } while (hasTablesToReplicate);

        return new SimpleResponse();
    }
}
