package ru.yandex.webmaster3.coordinator.http.mirrors;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.Setter;
import lombok.Value;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.storage.mirrors.dao.MainMirrorRequestsYDao;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequest;
import ru.yandex.webmaster3.storage.mirrors.data.MirrorRequestStateEnum;


/**
 * @author kravchenko99
 * @date 11/6/20
 */

@Component("/tmp/repeat/mirror/requests")
@RequiredArgsConstructor(onConstructor_ = @Autowired)
public class RepeatMirrorsRequestsAction extends Action<RepeatMirrorsRequestsAction.Req, RepeatMirrorsRequestsAction.Res> {

    private final MainMirrorRequestsYDao mainMirrorRequestsYDao;
    public static final Duration TTL = Duration.standardDays(90);


    @Override
    public Res process(Req request) throws WebmasterException {
        final DateTime dateFromRepeat = new DateTime(request.year, request.month, request.day, 0, 0);
        final List<MirrorRequest> mirrorRequests = mainMirrorRequestsYDao.listAllRequests()
                .stream()
                .filter(mirrorRequest -> mirrorRequest.getCreateDate().isAfter(dateFromRepeat))
                .collect(Collectors.toList());

        final Map<WebmasterHostId, List<UUID>> collect = new HashMap<>();
        // так как ChangeMainMirrorPeriodicTask срабатывает на таски которые обновились сутки назад
        final DateTime oneDayAgo = DateTime.now().minus(Duration.standardDays(1));
        for (var mirrorRequest : mirrorRequests) {
            collect.computeIfAbsent(mirrorRequest.getHostId(), ignore -> new ArrayList<>()).add(mirrorRequest.getRequestId());

            if (!request.dry) {
                final MirrorRequest newMainMirror = mirrorRequest
                        .withState(MirrorRequestStateEnum.NEW)
                        .withAttempts(0)
                        .withServiceResponse(null)
                        .withUpdateDate(oneDayAgo);
                mainMirrorRequestsYDao.saveRequest(newMainMirror);
            }
        }

        return new Res(collect, collect.size(), mirrorRequests.size());
    }

    @Setter(onMethod_ = @RequestQueryProperty(required = true))
    public static class Req implements ActionRequest {
        int year;
        int month;
        int day;
        boolean dry;
    }

    @Value
    public static class Res implements ActionResponse.NormalResponse {
        Map<WebmasterHostId, List<UUID>> repeatedRequests;
        int hostsCount;
        int requestsCount;
    }
}
