package ru.yandex.webmaster3.core.addurl;

import java.net.URI;
import java.util.UUID;

import javax.annotation.Nullable;

import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.RequiredArgsConstructor;
import lombok.ToString;
import lombok.With;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.joda.time.DateTime;
import org.joda.time.Duration;

import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.wmtools.common.util.uri.WebmasterUriUtils;

/**
 * @author tsyplyaev
 */
@RequiredArgsConstructor
@Getter
@EqualsAndHashCode(onlyExplicitlyIncluded = true)
@ToString
@With
public class UrlForRecrawl {
    public static final Duration STALE_REQUEST_AGE = Duration.standardDays(3);

    @EqualsAndHashCode.Include
    private final WebmasterHostId hostId;
    @EqualsAndHashCode.Include
    private final UUID urlId;
    private final String relativeUrl;

    private final DateTime addDate;
    private final DateTime processedDate;
    private final RecrawlState state;

    @JsonIgnore
    public String getRelativeUrl() {
        return relativeUrl;
    }

    public String getFullUrl() {
        return IdUtils.hostIdToReadableUrl(hostId) + relativeUrl;
    }

    public UrlForRecrawl changeState(RecrawlState s, DateTime procesedDate) {
        return new UrlForRecrawl(hostId, urlId, relativeUrl, addDate, procesedDate, s);
    }

    @Nullable
    public static Pair<WebmasterHostId, String> toHostIdAndRelativeUrl(String url) throws Exception {
        url = StringUtils.trimToEmpty(url);
        if (url.length() > 1024) {
            return null;
        }
        if (url.startsWith("/")) {
            return null;
        }

        URI uri = WebmasterUriUtils.toOldUri(url);
        WebmasterHostId hostIdFromUrl = IdUtils.urlToHostId(uri.toString());

        StringBuilder sb = new StringBuilder();
        if (StringUtils.isEmpty(uri.getPath())) {
            sb.append('/');
        } else {
            sb.append(uri.getPath());
        }

        if (uri.getQuery() != null) {
            sb.append('?').append(uri.getQuery());
        }

        return Pair.of(hostIdFromUrl, sb.toString());
    }

    public UrlForRecrawl withStaleCheck() {
        UrlForRecrawl r;
        if (!state.isTerminal() && addDate.isBefore(DateTime.now().minus(STALE_REQUEST_AGE))) {
            r = withState(RecrawlState.STALE).withProcessedDate(addDate.plus(STALE_REQUEST_AGE));
        } else {
            r = this;
        }

        return r;
    }
}
