package ru.yandex.webmaster3.core.blackbox.service;

import java.io.IOException;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.EnumMap;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.core.TreeNode;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import org.apache.commons.io.IOUtils;
import org.apache.http.NameValuePair;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.entity.UrlEncodedFormEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.client.utils.URIBuilder;
import org.apache.http.config.SocketConfig;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.message.BasicNameValuePair;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.blackbox.BlackboxAttributeType;
import ru.yandex.webmaster3.core.blackbox.BlackboxQuery;
import ru.yandex.webmaster3.core.blackbox.GetEmailsType;
import ru.yandex.webmaster3.core.blackbox.OAuthClientInfo;
import ru.yandex.webmaster3.core.blackbox.UserEmailInfo;
import ru.yandex.webmaster3.core.blackbox.UserWithFields;
import ru.yandex.webmaster3.core.blackbox.UserWithLogin;
import ru.yandex.webmaster3.core.data.L10nEnum;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;

/**
 * @author avhaliullin
 */
public class BlackboxUsersService extends AbstractExternalAPIService {
    private static final Logger log = LoggerFactory.getLogger(BlackboxUsersService.class);
    private static final ObjectMapper OM = new ObjectMapper();

    private static final int SOCKET_TIMEOUT = 2_000;


    private final URI blackboxAddress;
    private final TVMTokenService tvmTokenService;

    private CloseableHttpClient httpClient;

    public BlackboxUsersService(URI blackboxAddress,
                                TVMTokenService tvmTokenService) {
        this.blackboxAddress = blackboxAddress;
        this.tvmTokenService = tvmTokenService;
    }

    public void init() {
        SocketConfig socketConfig = SocketConfig.custom()
                .setSoTimeout(SOCKET_TIMEOUT)
                .build();

        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT)
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .build();

        httpClient = HttpClientBuilder.create()
                .setDefaultSocketConfig(socketConfig)
                .setMaxConnPerRoute(5)
                .setMaxConnTotal(5)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    public void destroy() {
        IOUtils.closeQuietly(httpClient);
    }

    public String getDefaultEmail(long uid) {
        List<UserEmailInfo> emails = getUserEmails(uid, GetEmailsType.DEFAULT);
        for (UserEmailInfo emailInfo : emails) {
            if (emailInfo.isValidatedEmail() && emailInfo.isDefaultEmail()) {
                return emailInfo.getEmail();
            }
        }
        return null;
    }

    public List<String> getValidatedEmails(long uid) {
        List<UserEmailInfo> emails = getUserEmails(uid, GetEmailsType.ALL);
        List<String> result = new ArrayList<>();
        for (UserEmailInfo emailInfo : emails) {
            if (emailInfo.isValidatedEmail()) {
                result.add(emailInfo.getEmail());
            }
        }
        return result;
    }

    public List<UserEmailInfo> getUserEmails(long uid, GetEmailsType getEmailsType) {
        List<UserWithFields> userWithFields = findUsers(BlackboxQuery.byUids(uid), Collections.emptySet(), getEmailsType);
        if (userWithFields.isEmpty()) {
            return Collections.emptyList();
        } else {
            return userWithFields.get(0).getEmails();
        }
    }

    public boolean isEmailValid(long uid, String email) {
        List<UserEmailInfo> emails = getUserEmails(uid, GetEmailsType.testOne(email));
        return emails.stream().anyMatch(UserEmailInfo::isValidatedEmail);
    }

    public Optional<String> getNormalizedUserEmail(long uid, String email) {
        List<UserEmailInfo> emails = getUserEmails(uid, GetEmailsType.testOne(email));
        return emails.stream().filter(UserEmailInfo::isValidatedEmail).map(UserEmailInfo::getEmail).findFirst();
    }

    @Nullable
    public UserWithLogin getUserById(long uid) {
        return getUserWithLogin(BlackboxQuery.byUids(uid));
    }

    @Nullable
    public UserWithLogin getUserByLogin(String login) {
        return getUserWithLogin(BlackboxQuery.byLogin(login));
    }

    public Optional<L10nEnum> getUserLanguage(long uid) {
        List<UserWithFields> userWithFieldsList = findUsers(
                BlackboxQuery.byUids(uid),
                EnumSet.of(BlackboxAttributeType.LANG),
                GetEmailsType.NONE
        );
        if (userWithFieldsList.isEmpty()) {
            return Optional.empty();
        }
        L10nEnum language;
        String langTag = userWithFieldsList.get(0).getFields().get(BlackboxAttributeType.LANG);
        switch (langTag) {
            case "en":
                language = L10nEnum.EN;
                break;
            case "tr":
                language = L10nEnum.TR;
                break;
            case "ru":
                language = L10nEnum.RU;
                break;
            case "kk":
                language = L10nEnum.KK;
                break;
            case "uk":
                language = L10nEnum.UK;
                break;
            case "be":
                language = L10nEnum.BE;
                break;
            default:
                return Optional.empty();
        }

        return Optional.of(language);
    }

    public Map<Long, String> mapUserIdsToLogins(Collection<Long> userIds) {
        List<UserWithFields> userWithFieldsList =
                findUsers(
                        BlackboxQuery.byUids(userIds),
                        EnumSet.of(BlackboxAttributeType.LOGIN),
                        GetEmailsType.NONE
                );
        return userWithFieldsList
                .stream()
                .filter(u -> u.getFields().containsKey(BlackboxAttributeType.LOGIN))
                .collect(Collectors.toMap(UserWithFields::getUid, u -> u.getFields().get(BlackboxAttributeType.LOGIN)));
    }

    @ExternalDependencyMethod("userinfo")
    public List<UserWithFields> findUsers(BlackboxQuery query, Set<BlackboxAttributeType> attributes, GetEmailsType getEmailsType) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            List<NameValuePair> params = new ArrayList<>();
            params.add(new BasicNameValuePair("method", "userinfo"));
            params.add(new BasicNameValuePair(query.param, query.value));
            fillEmailsParameters(getEmailsType, params);
            if (!attributes.isEmpty()) {
                params.add(new BasicNameValuePair("attributes",
                        attributes.stream().map(at -> Integer.toString(at.getId())).collect(Collectors.joining(","))));

            }
            JsonNode root = null;
            HttpPost request = createRequest(params);
            log.info("Querying blackbox with {}", request);
            try (CloseableHttpResponse response = httpClient.execute(request)) {
                root = processResponse(request, response);
                ObjectNode rootObject = asType(root, ObjectNode.class, true);
                ArrayNode usersArray = asType(rootObject.get("users"), ArrayNode.class, true);
                List<UserWithFields> result = new ArrayList<>();

                for (int i = 0; i < usersArray.size(); i++) {
                    ObjectNode userObject = asType(usersArray.get(i), ObjectNode.class, true);

                    String uidString = userObject.has("id") ? userObject.get("id").asText() : null;
                    if (uidString != null && userObject.has("login")) {
                        Map<BlackboxAttributeType, String> attrs = new EnumMap<>(BlackboxAttributeType.class);
                        JsonNode attrsNode = userObject.get("attributes");
                        if (attrsNode != null) {
                            for (BlackboxAttributeType attributeType : attributes) {
                                String attrName = String.valueOf(attributeType.getId());
                                if (attrsNode.has(attrName)) {
                                    String attrValue = attrsNode.get(attrName).asText();
                                    attrs.put(attributeType, attrValue);
                                }
                            }
                        }

                        long uid = Long.parseLong(uidString);

                        List<UserEmailInfo> emails = new ArrayList<>();
                        if (userObject.has("address-list")) {
                            for (JsonNode address : userObject.get("address-list")) {
                                emails.add(new UserEmailInfo(
                                        address.get("address").asText(),
                                        address.get("native").asBoolean(),
                                        address.get("default").asBoolean(),
                                        address.get("validated").asBoolean()
                                ));
                            }
                        }

                        result.add(new UserWithFields(uid, attrs, emails));
                    }
                }
                return result;
            } catch (IOException e) {
                throw new WebmasterException("Blackbox service failed, request = " + request.toString(),
                        new WebmasterErrorResponse.BlackboxErrorResponse(getClass(), e), e);
            } catch (WebmasterException e) {
                log.error("Error while processing blackbox response {}", root);
                throw e;
            }
        });
    }

    private void fillEmailsParameters(GetEmailsType getEmailsType, List<NameValuePair> params) {
        if (getEmailsType == GetEmailsType.NONE) {
            // do nothing
        } else if (getEmailsType == GetEmailsType.ALL) {
            params.add(new BasicNameValuePair("emails", "getall"));
        } else if (getEmailsType == GetEmailsType.DEFAULT) {
            params.add(new BasicNameValuePair("emails", "getdefault"));
        } else if (getEmailsType instanceof GetEmailsType.TestOne) {
            GetEmailsType.TestOne testOne = (GetEmailsType.TestOne) getEmailsType;
            params.add(new BasicNameValuePair("emails", "testone"));
            params.add(new BasicNameValuePair("addrtotest", testOne.getEmailToTest()));
        } else {
            throw new RuntimeException("Unknown get emails type " + getEmailsType);
        }
    }

    @ExternalDependencyMethod("oauth")
    public OAuthClientInfo checkOAuthToken(String token) {
        return trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            List<NameValuePair> params = new ArrayList<>();
            params.add(new BasicNameValuePair("method", "oauth"));
            params.add(new BasicNameValuePair("oauth_token", token));
            HttpPost request = createRequest(params);
            try (CloseableHttpResponse response = httpClient.execute(request)) {
                JsonNode root = processResponse(request, response);
                if (!"VALID".equals(root.get("status").get("value").asText())) {
                    return null;
                }
                long uid = Long.parseLong(root.get("uid").get("value").asText());
                JsonNode oauthNode = root.get("oauth");
                String clientName = oauthNode.get("client_name").asText();
                String clientId = oauthNode.get("client_id").asText();
                Set<OAuthClientInfo.Scope> scopes = new HashSet<>();
                String scopeString = oauthNode.get("scope").asText();
                String[] scopeStrings = scopeString.split(" ");
                for (String scopeName : scopeStrings) {
                    scopeName = scopeName.trim();
                    if (!scopeName.isEmpty()) {
                        String[] scopeParts = scopeName.split(":");
                        scopes.add(new OAuthClientInfo.Scope(scopeParts[0], scopeParts[1]));
                    }
                }

                return new OAuthClientInfo(uid, clientName, clientId, scopes);
            } catch (IOException e) {
                throw new WebmasterException("Blackbox service failed, request = " + request.toString(),
                        new WebmasterErrorResponse.BlackboxErrorResponse(getClass(), e), e);
            }
        });
    }

    //TODO: remove with webmaster3-download
    public <T> BlackboxResponse<T> execute(BlackboxRequest<T> request) {
        URIBuilder uriBuilder = new URIBuilder(blackboxAddress);
        uriBuilder = uriBuilder.setParameter("format", "json");

        HttpUriRequest httpRequest;
        try {
            httpRequest = request.createRequest(uriBuilder);
        } catch (URISyntaxException e) {
            log.error("Unable to create Blackbox request", e);
            return BlackboxResponse.createError(BlackboxStatus.REQUEST_ERROR);
        }
        log.info("Req: {}", httpRequest.getURI());

        try (CloseableHttpResponse httpResponse = httpClient.execute(httpRequest)) {
            return request.parseResponse(httpResponse);
        } catch (IOException e) {
            log.error("Unable to parse Blackbox response", e);
            return BlackboxResponse.createError(BlackboxStatus.REQUEST_ERROR);
        }
    }

    private HttpPost createRequest(List<NameValuePair> params) {
        HttpPost request = new HttpPost(blackboxAddress);
        params.add(new BasicNameValuePair("userip", "127.0.0.1"));
        params.add(new BasicNameValuePair("format", "json"));
        if (tvmTokenService != null) {
            final String token = tvmTokenService.getToken();
            params.add(new BasicNameValuePair(TVMTokenService.TVM2_TICKET_HEADER, token));
            request.addHeader(TVMTokenService.TVM2_TICKET_HEADER, token);
        }
        request.setEntity(new UrlEncodedFormEntity(params, StandardCharsets.UTF_8));
        return request;
    }

    private static JsonNode processResponse(HttpPost request, CloseableHttpResponse response) throws IOException {
        if (response.getStatusLine().getStatusCode() != 200) {
            throw new WebmasterException("Blackbox service returned code " + response.getStatusLine().getStatusCode() +
                    ", request = " + request.toString(),
                    new WebmasterErrorResponse.BlackboxErrorResponse(BlackboxUsersService.class, null));
        }
        return OM.readTree(response.getEntity().getContent());
    }

    @Nullable
    private UserWithLogin getUserWithLogin(BlackboxQuery query) {
        List<UserWithFields> users = findUsers(query, EnumSet.of(BlackboxAttributeType.LOGIN), GetEmailsType.NONE);
        if (users.isEmpty()) {
            return null;
        }
        UserWithFields userWithFields = users.get(0);
        return new UserWithLogin(userWithFields.getUid(), userWithFields.getFields().get(BlackboxAttributeType.LOGIN));
    }

    private <T extends TreeNode> T asType(TreeNode node, Class<T> clazz, boolean forceNotNull) {
        if (node == null) {
            if (forceNotNull) {
                throw new WebmasterException("Failed to process blackbox response, expected " + clazz.getSimpleName() + " found null",
                        new WebmasterErrorResponse.BlackboxErrorResponse(getClass(), null));
            } else {
                return null;
            }
        }
        if (clazz.isAssignableFrom(node.getClass())) {
            return (T) node;
        } else {
            throw new WebmasterException("Failed to process blackbox response, expected " + clazz.getSimpleName() + " found " + node,
                    new WebmasterErrorResponse.BlackboxErrorResponse(getClass(), null));
        }
    }
}
