package ru.yandex.webmaster3.core.checklist.data;

import java.util.ArrayList;
import java.util.EnumSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import com.fasterxml.jackson.annotation.JsonProperty;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Getter;
import org.apache.commons.collections4.CollectionUtils;
import org.joda.time.DateTime;
import org.joda.time.format.DateTimeFormat;

import ru.yandex.autodoc.common.doc.annotation.Description;
import ru.yandex.webmaster3.core.util.enums.IntEnum;
import ru.yandex.webmaster3.core.util.enums.IntEnumResolver;
import ru.yandex.webmaster3.core.data.HttpCodeInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.host.SlowUrlSampleInfo;
import ru.yandex.webmaster3.core.hoststat.HostStatisticsInfo.ContentAttrSampleInfo;
import ru.yandex.webmaster3.core.hoststat.HostStatisticsInfo.PageSampleInfo;
import ru.yandex.webmaster3.core.http.HierarchyTypeField;
import ru.yandex.webmaster3.core.sitemap.SitemapInfo;
import ru.yandex.webmaster3.core.turbo.model.ban.TurboBanReason;
import ru.yandex.webmaster3.core.turbo.model.feed.TurboFeedInfo;
import ru.yandex.webmaster3.core.util.TimeUtils;
import ru.yandex.webmaster3.core.util.json.polymorphic.Polymorphic;
import ru.yandex.wmtools.common.util.http.YandexHttpStatus;

/**
 * @author avhaliullin
 */
@JsonTypeInfo(use = JsonTypeInfo.Id.CUSTOM, include = JsonTypeInfo.As.EXISTING_PROPERTY, property = "problemType")
public abstract class SiteProblemContent implements Polymorphic<SiteProblemTypeEnum> {
    private final SiteProblemTypeEnum problemType;

    SiteProblemContent(SiteProblemTypeEnum problemType) {
        this.problemType = problemType;
    }

    /**
     * Объединяет проблемы одного типа - если данные об одной проблеме могут быть получены из разных источников
     *
     * @param that - проблема, с которой нужно смержить текущую
     * @return - объединенная проблема
     * @throws IllegalArgumentException - если переданный объект имеет другой тип
     */
    public abstract SiteProblemContent merge(SiteProblemContent that);

    @HierarchyTypeField
    public SiteProblemTypeEnum getProblemType() {
        return problemType;
    }

    @Override
    @JsonProperty(value = "problemType")
    public SiteProblemTypeEnum getType() {
        return problemType;
    }

    @JsonIgnore
    public boolean isRecheckable() {
        return true;
    }

    @JsonIgnore
    public boolean hasSamples() {
        return false;
    }

    public static abstract class AbstractNoDataProblem extends SiteProblemContent {
        protected AbstractNoDataProblem(SiteProblemTypeEnum problemType) {
            super(problemType);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("К сайту применены санкции")
    public static class SanctionsProblem extends SiteProblemContent {
        private final WebmasterHostId ownerHostId;

        @JsonCreator
        public SanctionsProblem(@JsonProperty("ownerHostId") WebmasterHostId ownerHostId) {
            super(SiteProblemTypeEnum.SANCTIONS);
            this.ownerHostId = ownerHostId;
        }

        @Description("owner, для которого приехали санкции. Если null - то совпадает с id хоста")
        public WebmasterHostId getOwnerHostId() {
            return ownerHostId;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Проблемы безопасности")
    public static class SecurityProblem extends AbstractNoDataProblem {
        public SecurityProblem() {
            super(SiteProblemTypeEnum.SECURITY);
        }
    }

    @Description("Нет сайтмепов")
    public static class SitemapNotSet extends AbstractNoDataProblem {
        public SitemapNotSet() {
            super(SiteProblemTypeEnum.SITEMAP_NOT_SET);
        }
    }

    @JsonIgnoreProperties(ignoreUnknown = true)
    @Description("Сайтмепы содержат ошибки")
    public static class ErrorsInSitemaps extends AbstractNoDataProblem {
        private final List<SitemapInfo> sitemaps;

        @JsonCreator
        public ErrorsInSitemaps(@JsonProperty("sitemaps") List<SitemapInfo> sitemaps) {
            super(SiteProblemTypeEnum.ERRORS_IN_SITEMAPS);
            this.sitemaps = sitemaps;
        }

        public List<SitemapInfo> getSitemaps() {
            return sitemaps;
        }
    }

    @Description("Содержимое файлов sitemap давно не менялось")
    public static class NoSitemapModifications extends SiteProblemContent {
        private final DateTime lastModification;

        @JsonCreator
        public NoSitemapModifications(@JsonProperty("lastModification") DateTime lastModification) {
            super(SiteProblemTypeEnum.NO_SITEMAP_MODIFICATIONS);
            this.lastModification = lastModification;
        }

        public DateTime getLastModification() {
            return lastModification;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            if (!(that instanceof NoSitemapModifications)) {
                throw new IllegalArgumentException("Expected " + getClass() + " but got " + that);
            }
            return TimeUtils.latestBy(NoSitemapModifications::getLastModification, this, (NoSitemapModifications) that);
        }
    }

    @Description("Отсутствуют регионы сайта")
    public static class NoRegions extends AbstractNoDataProblem {
        public NoRegions() {
            super(SiteProblemTypeEnum.NO_REGIONS);
        }
    }

    @Description("Отсутствуют регионы из справочника (сайт не зарегистрирован в справочнике)")
    public static class NoDictionaryRegions extends AbstractNoDataProblem {
        public NoDictionaryRegions() {
            super(SiteProblemTypeEnum.NO_DICTIONARY_REGIONS);
        }
    }

    @Description("Отсутствуют быстрые ссылки")
    public static class NoSerpLinks extends AbstractNoDataProblem {
        public NoSerpLinks() {
            super(SiteProblemTypeEnum.NO_SERPLINKS);
        }
    }

    @Description("Сайт запрещен к индексированию в robots.txt")
    public static class DisallowedInRobots extends AbstractNoDataProblem {
        public DisallowedInRobots() {
            super(SiteProblemTypeEnum.DISALLOWED_IN_ROBOTS);
        }
    }

    @Description("Возникла ошибка DNS при попытке индексирования сайта")
    public static class DNSError extends AbstractNoDataProblem {
        public DNSError() {
            super(SiteProblemTypeEnum.DNS_ERROR);
        }
    }

    @Description("Возникла ошибка при установки связи с сервером")
    public static class ConnectFailed extends AbstractNoDataProblem {
        public ConnectFailed() {
            super(SiteProblemTypeEnum.CONNECT_FAILED);
        }
    }

    @Description("На сайте возможно отсутствует поддержка 404 ошибок")
    public static class No404Errors extends AbstractNoDataProblem {
        public No404Errors() {
            super(SiteProblemTypeEnum.NO_404_ERRORS);
        }
    }

    @Description("На сайте проблема с robots.txt")
    public static class RobotsTxtError extends AbstractNoDataProblem {
        public RobotsTxtError() {
            super(SiteProblemTypeEnum.ROBOTS_TXT_ERROR);
        }
    }

    @Description("Срок действия доменного имени подходит к концу")
    public static class DomainNameExpires extends SiteProblemContent {
        private final DateTime parseDate;
        private final DateTime expirationDate;

        @JsonCreator
        public DomainNameExpires(@JsonProperty("parseDate") DateTime parseDate, @JsonProperty("expirationDate") DateTime expirationDate) {
            super(SiteProblemTypeEnum.DOMAIN_EXPIRES);
            this.parseDate = parseDate;
            this.expirationDate = expirationDate;
        }

        public DateTime getParseDate() {
            return parseDate;
        }

        public DateTime getExpirationDate() {
            return expirationDate;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            if (!(that instanceof DomainNameExpires)) {
                throw new IllegalArgumentException("Expected " + getClass() + " but got " + that);
            }
            return TimeUtils.latestBy(DomainNameExpires::getParseDate, this, (DomainNameExpires) that);
        }
    }

    @Description("Главная страница сайта возвращает ошибку")
    public static class MordaError extends SiteProblemContent {
        private final DateTime lastAccess;
        private final ExtendedStatus extendedStatus;
        private final YandexHttpStatus httpStatus;
        private final boolean followedRedirect;

        @JsonCreator
        public MordaError(
                @JsonProperty("lastAccess") DateTime lastAccess,
                @JsonProperty("extendedStatus") ExtendedStatus extendedStatus,
                @JsonProperty("httpStatus") YandexHttpStatus httpStatus,
                @JsonProperty("followedRedirect") boolean followedRedirect) {
            super(SiteProblemTypeEnum.MORDA_ERROR);
            this.lastAccess = lastAccess;
            this.extendedStatus = extendedStatus;
            this.httpStatus = httpStatus;
            this.followedRedirect = followedRedirect;
        }

        public DateTime getLastAccess() {
            return lastAccess;
        }

        public YandexHttpStatus getHttpStatus() {
            return httpStatus;
        }

        public ExtendedStatus getExtendedStatus() {
            return extendedStatus;
        }

        public HttpCodeInfo getHttpCodeInfo() {
            return httpStatus == null ? null : HttpCodeInfo.fromHttpStatus(httpStatus);
        }

        public Integer getHttpStatusCode() {
            return httpStatus == null ? null : httpStatus.getCode();
        }

        public boolean isFollowedRedirect() {
            return followedRedirect;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            if (!(that instanceof MordaError)) {
                throw new IllegalArgumentException("Expected " + getClass() + " but got " + that);
            }
            return TimeUtils.latestBy(MordaError::getLastAccess, this, (MordaError) that);
        }

        public enum ExtendedStatus implements IntEnum {
            @Description("Статус следует смотреть в httpStatus")
            DEFAULT(0),
            @Description("Циклический редирект")
            REDIRECT_LOOP(1),
            @Description("URL отсутствует в киви")
            URL_NOT_FOUND(2),
            ;
            private final int value;

            ExtendedStatus(int value) {
                this.value = value;
            }

            @Override
            public int value() {
                return value;
            }

            public static IntEnumResolver<ExtendedStatus> R = IntEnumResolver.r(ExtendedStatus.class);
        }
    }

    @Description("Главная страница сайта редиректит на другой хост")
    public static class MordaRedirects extends SiteProblemContent {
        private final DateTime lastAccess;
        private final String targetHost;

        @JsonCreator
        public MordaRedirects(@JsonProperty("lastAccess") DateTime lastAccess, @JsonProperty("targetHost") String targetHost) {
            super(SiteProblemTypeEnum.MORDA_REDIRECTS);
            this.lastAccess = lastAccess;
            this.targetHost = targetHost;
        }

        public DateTime getLastAccess() {
            return lastAccess;
        }

        public String getTargetHost() {
            return targetHost;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            if (!(that instanceof MordaRedirects)) {
                throw new IllegalArgumentException("Expected " + getClass() + " but got " + that);
            }
            return TimeUtils.latestBy(MordaRedirects::getLastAccess, this, (MordaRedirects) that);
        }
    }

    @Description("У сайта нет фавиконки")
    public static class MissingFavicon extends AbstractNoDataProblem {
        public MissingFavicon() {
            super(SiteProblemTypeEnum.MISSING_FAVICON);
        }
    }

    public static class TooManyUrlDuplicates extends SiteProblemContent {
        private final DateTime lastUpdate;
        private final long count;

        @JsonCreator
        public TooManyUrlDuplicates(@JsonProperty("lastUpdate") DateTime lastUpdate, @JsonProperty("count") long count) {
            super(SiteProblemTypeEnum.TOO_MANY_URL_DUPLICATES);
            this.lastUpdate = lastUpdate;
            this.count = count;
        }

        public DateTime getLastUpdate() {
            return lastUpdate;
        }

        public long getCount() {
            return count;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            if (!(that instanceof TooManyUrlDuplicates)) {
                throw new IllegalArgumentException("Expected " + getClass() + " but got " + that);
            }
            return TimeUtils.latestBy(TooManyUrlDuplicates::getLastUpdate, this, (TooManyUrlDuplicates) that);
        }
    }

    @Description("У сайта нет robots.txt (не удалось скачать)")
    public static class NoRobotsTxt extends AbstractNoDataProblem {
        public NoRobotsTxt() {
            super(SiteProblemTypeEnum.NO_ROBOTS_TXT);
        }
    }

    @Description("Не указана директива Host")
    public static class NoHostDirective extends AbstractNoDataProblem {
        public NoHostDirective() {
            super(SiteProblemTypeEnum.NO_HOST_DIRECTIVE);
        }
    }

    @Description("Противоречивые директивы Host")
    public static class AmbiguousHostDirectives extends AbstractNoDataProblem {
        public AmbiguousHostDirectives() {
            super(SiteProblemTypeEnum.AMBIGUOUS_HOST_DIRECTIVES);
        }
    }

//    @Description("Ошибки в микроразметке")
//    public static class MicrodataErrors extends AbstractNoDataProblem {
//        public MicrodataErrors() {
//            super(SiteProblemTypeEnum.MICRODATA_ERRORS);
//        }
//    }

    @Description("Большое количество неработающих внутренних ссылок")
    public static class TooManyBrokenLinks extends AbstractNoDataProblem {
        public TooManyBrokenLinks() {
            super(SiteProblemTypeEnum.TOO_MANY_BROKEN_LINKS);
        }
    }

    @Getter
    @Description("Сайт не мобилопригоден")
    public static class NotMobileFriendly extends SiteProblemContent {
        private final MobileAuditResolution resolution;
        private final int notMobileFriendlyStatus;

        public NotMobileFriendly(@JsonProperty("resolution") MobileAuditResolution resolution,
                                 @JsonProperty("notMobileFriendlyStatus") int notMobileFriendlyStatus) {
            super(SiteProblemTypeEnum.NOT_MOBILE_FRIENDLY);
            this.resolution = resolution;
            this.notMobileFriendlyStatus = notMobileFriendlyStatus;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Много документов с пустым title")
    public static class DocumentsMissingTitle extends AbstractNoDataProblem {
        public DocumentsMissingTitle() {
            super(SiteProblemTypeEnum.DOCUMENTS_MISSING_TITLE);
        }
    }

    @Description("Много документов с пустым description")
    public static class DocumentsMissingDescription extends AbstractNoDataProblem {
        public DocumentsMissingDescription() {
            super(SiteProblemTypeEnum.DOCUMENTS_MISSING_DESCRIPTION);
        }
    }

    @Description("Слишком большое среднее время ответа")
    public static class SlowResponse extends SiteProblemContent {
        private final long avgResponseTime;

        public SlowResponse(@JsonProperty("avgResponseTime") long avgResponseTime) {
            super(SiteProblemTypeEnum.SLOW_AVG_RESPONSE);
            this.avgResponseTime = avgResponseTime;
        }

        public long getAvgResponseTime() {
            return avgResponseTime;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Слишком большое среднее время ответа")
    @Getter
    public static class SlowResponseWithExamples extends SiteProblemContent {
        private final long avgResponseTime;
        private final List<SlowUrlSampleInfo> samples;

        public SlowResponseWithExamples(@JsonProperty("avgResponseTime") long avgResponseTime,
                                        @JsonProperty("samples") List<SlowUrlSampleInfo> samples) {
            super(SiteProblemTypeEnum.SLOW_AVG_RESPONSE_WITH_EXAMPLES);
            this.avgResponseTime = avgResponseTime;
            this.samples = samples.stream().limit(5).collect(Collectors.toList());
        }

        @JsonIgnore
        public String searializeSamplesAsHtml() {
            return samples.stream()
                    .map(x -> x.getPath() + " " + new DateTime(x.getLastAccess() * 1000).toString(DateTimeFormat.forPattern("yyyy-MM-dd HH:mm:ss z").withZoneUTC()) + " " + x.getFetchTime() + "ms")
                    .map(x -> String.format("<li>%s</li>", x))
                    .collect(Collectors.joining("\n"));
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("На сайте обнаружены нарушения")
    public static class Threats extends AbstractNoDataProblem {
        public Threats() {
            super(SiteProblemTypeEnum.THREATS);
        }
    }

    public static class NoMetrikaCounter extends AbstractNoDataProblem {
        public NoMetrikaCounter() {
            super(SiteProblemTypeEnum.NO_METRIKA_COUNTER);
        }
    }

    public static final class TurboInsufficientClicksShare extends SiteProblemContent {

        private final int sharePercentage;

        public TurboInsufficientClicksShare(@JsonProperty("sharePercentage") int sharePercentage) {
            super(SiteProblemTypeEnum.TURBO_INSUFFICIENT_CLICKS_SHARE);
            this.sharePercentage = sharePercentage;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        @Description("Доля турбо-кликов среди всех мобильных кликов за неделю, в процентах")
        public int getSharePercentage() {
            return sharePercentage;
        }
    }

    public static class TurboProblemContent extends SiteProblemContent {

        private final List<TurboFeedInfo> feeds;

        protected TurboProblemContent(SiteProblemTypeEnum problemType, List<TurboFeedInfo> feeds) {
            super(problemType);
            this.feeds = feeds;
        }

        @Description("Фиды с соответствующей проблемой")
        public List<TurboFeedInfo> getFeeds() {
            return feeds;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    public static class TurboError extends TurboProblemContent {
        public TurboError(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_ERROR, feeds);
        }
    }

    public static class TurboWarning extends TurboProblemContent {
        public TurboWarning(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_WARNING, feeds);
        }
    }

    @Description("Я.Браузер - плохая реклама")
    public static class YaBrowserBadAd extends SiteProblemContent {

        private final DateTime lastUpdate;
        private final Long filtrationStartMonth;
        private final DateTime warnExpirationDate;
        private final YaBrowserBadAdStatus status;
        private final EnumSet<YaBrowserBadAdFormat> formats;

        public YaBrowserBadAd(@JsonProperty("lastUpdate") DateTime lastUpdate,
                              @JsonProperty("filtrationStartMonth") Long filtrationStartMonth,
                              @JsonProperty("warnExpirationDate") DateTime warnExpirationDate,
                              @JsonProperty("status") YaBrowserBadAdStatus status,
                              @JsonProperty("formats") EnumSet<YaBrowserBadAdFormat> formats) {
            super(SiteProblemTypeEnum.YABROWSER_BADAD);
            this.lastUpdate = lastUpdate;
            this.filtrationStartMonth = filtrationStartMonth;
            this.warnExpirationDate = warnExpirationDate;
            this.status = status;
            this.formats = formats;
        }

        @Description("Дата последнего обновления информации о плохой рекламе")
        public DateTime getLastUpdate() {
            return lastUpdate;
        }

        @Description("С какого месяца начнется (или началась) фильтрация плохой рекламы [Поле уже не актуально]")
        public Long getFiltrationStartMonth() {
            return filtrationStartMonth;
        }

        @Description("Когда истекает дата предупреждения")
        public DateTime getWarnExpirationDate() {
            return warnExpirationDate;
        }

        @Description("Статус блокировки - активна или только предупреждение")
        public YaBrowserBadAdStatus getStatus() {
            return status;
        }

        @Description("Форматы плохой рекламы, присутствующей на данном сайте")
        public EnumSet<YaBrowserBadAdFormat> getFormats() {
            return formats;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        public enum YaBrowserBadAdStatus {
            ACTIVE,
            WARNING,
        }
    }

    @Getter
    public static abstract class TurboBanContent extends SiteProblemContent {

        @Description("true - если бан по домену")
        private final boolean banByDomain;
        @Description("Причины забанов")
        private final Set<TurboBanReason> reasons;
        @Description("С какой даты начнет применяться данный бан (если фидов несколько - то минимальная дата)")
        private final DateTime applyDate;
        @Description("Забаненные фиды")
        private final List<String> feeds;
        @JsonIgnore
        private final List<TurboBanSample> samples = new ArrayList<>();

        public TurboBanContent(SiteProblemTypeEnum problemType, boolean banByDomain, Set<TurboBanReason> reasons,
                               DateTime applyDate, List<String> feeds) {
            super(problemType);
            this.banByDomain = banByDomain;
            this.reasons = reasons;
            this.applyDate = applyDate;
            this.feeds = feeds;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        @Override
        public boolean hasSamples() {
            return !CollectionUtils.isEmpty(samples);
        }
    }

    @Description("Баны турбо-фидов")
    public static final class TurboFeedBan extends TurboBanContent {

        public TurboFeedBan(@JsonProperty("banByDomain") boolean banByDomain,
                            @JsonProperty("reasons") Set<TurboBanReason> reasons,
                            @JsonProperty("applyDate") DateTime applyDate,
                            @JsonProperty("feeds") List<String> feeds) {
            super(SiteProblemTypeEnum.TURBO_FEED_BAN, banByDomain, reasons, applyDate, feeds);
        }
    }

    @Description("Баны турбо-документов")
    public static final class TurboDocumentBan extends TurboBanContent {

        private final long count;

        public TurboDocumentBan(@JsonProperty("banByDomain") boolean banByDomain,
                                @JsonProperty("reasons") Set<TurboBanReason> reasons,
                                @JsonProperty("applyDate") DateTime applyDate,
                                @JsonProperty("count") long count,
                                @JsonProperty("feeds") List<String> feeds) {
            super(SiteProblemTypeEnum.TURBO_DOCUMENT_BAN, banByDomain, reasons, applyDate, feeds);
            this.count = count;
        }

        @Description("Сколько всего документов забанено")
        public long getCount() {
            return count;
        }
    }

    public static class TurboHostBan extends SiteProblemContent {

        private final TurboHostBanSeverity severity;
        private final Long deadlineTs;

        @JsonCreator
        public TurboHostBan(
                @JsonProperty("severity") TurboHostBanSeverity severity,
                @JsonProperty("deadlineTs") Long deadlineTs
        ) {
            super(SiteProblemTypeEnum.TURBO_HOST_BAN);
            this.severity = severity;
            this.deadlineTs = deadlineTs;
        }

        public TurboHostBan(SiteProblemTypeEnum problemType, TurboHostBanSeverity severity, Long deadlineTs) {
            super(problemType);
            this.severity = severity;
            this.deadlineTs = deadlineTs;
        }

        @Description("Критичность проблемы")
        @JsonProperty(value = "severity", defaultValue = "WARNING")
        public TurboHostBanSeverity getSeverity() {
            return severity;
        }

        @Description("Время дедлайна для WARNING")
        @JsonProperty(value = "deadlineTs")
        public Long getDeadlineTs() {
            return deadlineTs;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        @Override
        public boolean isRecheckable() {
            return severity == TurboHostBanSeverity.BAN
                    || severity == TurboHostBanSeverity.WARNING
                    || severity == TurboHostBanSeverity.INFO;
        }

        public enum TurboHostBanSeverity implements IntEnum {
            OK(0),
            INFO(1),
            WARNING(2),
            BAN(3);

            private final int value;

            TurboHostBanSeverity(int value) {
                this.value = value;
            }

            @Override
            public int value() {
                return value;
            }

            public static final IntEnumResolver<TurboHostBanSeverity> R = IntEnumResolver.r(TurboHostBanSeverity.class);
        }
    }


    public static final class TurboHostBanInfo extends TurboHostBan {

        @JsonCreator
        public TurboHostBanInfo(
                @JsonProperty("severity") TurboHostBanSeverity severity,
                @JsonProperty("deadlineTs") Long deadlineTs
        ) {
            super(SiteProblemTypeEnum.TURBO_HOST_BAN_INFO, severity, deadlineTs);
        }
    }

    public static final class TurboHostBanOk extends SiteProblemContent {

        public TurboHostBanOk() {
            super(SiteProblemTypeEnum.TURBO_HOST_BAN_OK);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    public static class TooManyDomainsOnSearch extends SiteProblemContent {

        public static final int THRESHOLD = 10;

        private final List<String> subdomains;

        public TooManyDomainsOnSearch(@JsonProperty("subdomains") List<String> subdomains) {
            super(SiteProblemTypeEnum.TOO_MANY_DOMAINS_ON_SEARCH);
            Preconditions.checkArgument(subdomains != null && !subdomains.isEmpty());
            this.subdomains = new ArrayList<>(subdomains.subList(0, Math.min(subdomains.size(), THRESHOLD)));
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        public List<String> getSubdomains() {
            return subdomains;
        }
    }

    @Description("Робот не смог проиндексировать видео, размеченные на сайте")
    public static class NoChats extends AbstractNoDataProblem {
        public NoChats() {
            super(SiteProblemTypeEnum.NO_CHATS);
        }
    }

    @Description("Есть незаполненные организации в Яндекс.Справочнике")
    public static class HostCompanyProfileNotFilled extends SiteProblemContent {
        private final WebmasterHostId hostId;
        private final List<CompanyInfo> companyInfos;

        public HostCompanyProfileNotFilled(@JsonProperty("hostId") WebmasterHostId hostId,
                                           @JsonProperty("companyInfos") List<CompanyInfo> companyInfos) {
            super(SiteProblemTypeEnum.HOST_COMPANY_PROFILE_NOT_FILLED);
            this.hostId = hostId;
            this.companyInfos = companyInfos;
        }

        public WebmasterHostId getHostId() {
            return hostId;
        }

        public List<CompanyInfo> getCompanyInfos() {
            return companyInfos;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Яндекс.Справочник создал карточку организации.")
    public static class HostCompanyProfileCreated extends SiteProblemContent {
        private final String tycoonLink;

        public HostCompanyProfileCreated(@JsonProperty("tycoonLink") String tycoonLink) {
            super(SiteProblemTypeEnum.HOST_COMPANY_PROFILE_CREATED);
            this.tycoonLink = tycoonLink;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        public String getTycoonLink() {
            return tycoonLink;
        }
    }

    @Description("Некорректная настройка SSL-сертификата сайта")
    public static class SslCertificateError extends SiteProblemContent {
        public SslCertificateError() {
            super(SiteProblemTypeEnum.SSL_CERTIFICATE_ERROR);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Главное зеркало сайта - не на HTTPS")
    public static class MainMirrorIsNotHttps extends SiteProblemContent {
        public MainMirrorIsNotHttps() {
            super(SiteProblemTypeEnum.MAIN_MIRROR_IS_NOT_HTTPS);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Проблема с фавиконкой/фавиконками")
    public static class FaviconError extends SiteProblemContent {

        public FaviconError() {
            super(SiteProblemTypeEnum.FAVICON_ERROR);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Проблемы с листингами турбо")
    public static class TurboListingError extends SiteProblemContent {
        public TurboListingError() {
            super(SiteProblemTypeEnum.TURBO_LISTING_ERROR);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Не включен обход ни по одному привязанному счетику Метрики")
    public static class NoMetrikaCounterCrawlEnabled extends SiteProblemContent {
        public NoMetrikaCounterCrawlEnabled() {
            super(SiteProblemTypeEnum.NO_METRIKA_COUNTER_CRAWL_ENABLED);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Getter
    @Description("Проблемы со страницами из турбо-фидов")
    public static class TurboUrlErrors extends SiteProblemContent {

        @Description("Сколько плохих турбо-страниц")
        private final long bad;
        @Description("Сколько всего турбо-страниц")
        private final long total;

        public TurboUrlErrors(@JsonProperty("bad") long bad, @JsonProperty("total") long total) {
            super(SiteProblemTypeEnum.TURBO_URL_ERRORS);
            this.bad = bad;
            this.total = total;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Нет провязок счетчиков Метрики")
    public static class NoMetrikaCounterBinding extends SiteProblemContent {
        public NoMetrikaCounterBinding() {
            super(SiteProblemTypeEnum.NO_METRIKA_COUNTER_BINDING);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Отсутствуют фавиконки больших размеров")
    public static class BigFaviconAbsent extends SiteProblemContent {
        public BigFaviconAbsent() {
            super(SiteProblemTypeEnum.BIG_FAVICON_ABSENT);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Getter
    @Description("Некорректный URL старой Турбо-корзины")
    public static class TurboInvalidCartUrl extends SiteProblemContent {

        @Description("Конкретный статус")
        private final CartUrlStatus status;

        public TurboInvalidCartUrl(@JsonProperty("status") CartUrlStatus status) {
            super(SiteProblemTypeEnum.TURBO_INVALID_CART_URL);
            this.status = status;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }

        @AllArgsConstructor
        @Getter
        public enum CartUrlStatus {
            @Description("Еще не проверялся или на проверке сейчас")
            NEW(false),
            @Description("Все ок")
            OK(false),
            @Description("Невозможно проверить, т.к. нет примеров в history_static")
            CHECK_UNAVAILABLE(false),
            @Description("Не удалось сделать скриншот, например из-за недоступности сервера/4хх/5хх ответа")
            ERROR_UNAVAILABLE(true),
            @Description("Не корзина")
            ERROR_NOT_CART(true),
            @Description("Корзина, но требуемый товар в нее не добавлен")
            ERROR_OFFER_NOT_ADDED(true),
            @Description("Нет уверенности в ответе")
            UNCERTAIN(false), // недостаточно достоверности
            @Description("Корзина отключена")
            OFF(false),
            ;

            private final boolean bad;
        }
    }

    public static class TurboRssError extends TurboProblemContent {
        public TurboRssError(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_RSS_ERROR, feeds);
        }
    }

    public static class TurboRssWarning extends TurboProblemContent {
        public TurboRssWarning(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_RSS_WARNING, feeds);
        }
    }

    public static class TurboYmlError extends TurboProblemContent {
        public TurboYmlError(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_YML_ERROR, feeds);
        }
    }

    public static class TurboYmlWarning extends TurboProblemContent {
        public TurboYmlWarning(@JsonProperty("feeds") List<TurboFeedInfo> feeds) {
            super(SiteProblemTypeEnum.TURBO_YML_WARNING, feeds);
        }
    }

    @Getter
    @Description("Найдены незначащие CGI параметры")
    public static class InsignificantCGIParameter extends SiteProblemContent {
        private final Map<String, List<String>> cgiParameters;

        @JsonCreator
        public InsignificantCGIParameter(@JsonProperty("cgiParameters") Map<String, List<String>> cgiParameters) {
            super(SiteProblemTypeEnum.INSIGNIFICANT_CGI_PARAMETER);
            this.cgiParameters = cgiParameters;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Getter
    @Description("Найдены дубли заголовков/дескрипшенов")
    public static class DuplicateContentAttrs extends SiteProblemContent {
        private final long duplicateTitles;
        private final double prevailingTitleShare;
        private final long duplicateDescriptions;
        private final double prevailingDescriptionShare;
        private final List<ContentAttrSampleInfo> prevailingTitles;
        private final List<ContentAttrSampleInfo> prevailingDescriptions;

        @JsonCreator
        public DuplicateContentAttrs(@JsonProperty("duplicateTitles") long duplicateTitles,
                                     @JsonProperty("prevailingTitleShare") double prevailingTitleShare,
                                     @JsonProperty("duplicateDescriptions") long duplicateDescriptions,
                                     @JsonProperty("prevailingDescriptionShare") double prevailingDescriptionShare,
                                     @JsonProperty("prevailingTitles") List<ContentAttrSampleInfo> prevailingTitles,
                                     @JsonProperty("prevailingDescriptions") List<ContentAttrSampleInfo> prevailingDescriptions) {
            super(SiteProblemTypeEnum.DUPLICATE_CONTENT_ATTRS);
            this.duplicateTitles = duplicateTitles;
            this.prevailingTitleShare = prevailingTitleShare;
            this.duplicateDescriptions = duplicateDescriptions;
            this.prevailingDescriptionShare = prevailingDescriptionShare;
            this.prevailingTitles = prevailingTitles;
            this.prevailingDescriptions = prevailingDescriptions;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Getter
    @Description("Найдены дубли страниц")
    public static class DuplicatePages extends SiteProblemContent {

        private final long duplicatePages;
        private final double duplicatePagesShare;
        private final List<PageSampleInfo> samples;

        @JsonCreator
        public DuplicatePages(@JsonProperty("duplicatePages") long duplicatePages,
                              @JsonProperty("duplicatePagesShare") double duplicatePagesShare,
                              @JsonProperty("samples") List<PageSampleInfo> samples) {
            super(SiteProblemTypeEnum.DUPLICATE_PAGES);
            this.duplicatePages = duplicatePages;
            this.duplicatePagesShare = duplicatePagesShare;
            this.samples = samples;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }


    @Getter
    @Description("todo")
    public static class UrlAlertXxx extends SiteProblemContent {
        private final long hourInterval;

        @JsonCreator
        public UrlAlertXxx(
                SiteProblemTypeEnum typeEnum,
                long hourInterval) {
            super(typeEnum);
            this.hourInterval = hourInterval;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("todo")
    public static class UrlAlert4xx extends UrlAlertXxx {

        @JsonCreator
        public UrlAlert4xx(@JsonProperty("hourInterval") long hourInterval) {
            super(SiteProblemTypeEnum.URL_ALERT_4XX, hourInterval);
        }
    }

    @Description("todo")
    public static class UrlAlert5xx extends UrlAlertXxx {
        @JsonCreator
        public UrlAlert5xx(@JsonProperty("hourInterval") long hourInterval) {
            super(SiteProblemTypeEnum.URL_ALERT_5XX, hourInterval);
        }
    }

    @Getter
    @Description("Найдены полезные страницы, закрытые от индексирования")
    public static class DisallowedUrlsAlert extends SiteProblemContent {
        @JsonCreator
        public DisallowedUrlsAlert() {
            super(SiteProblemTypeEnum.DISALLOWED_URLS_ALERT);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Требуется оферта для видеохостинга")
    public static class VideohostOfferIsNeeded extends SiteProblemContent {
        @JsonCreator
        public VideohostOfferIsNeeded() {
            super(SiteProblemTypeEnum.VIDEOHOST_OFFER_IS_NEEDED);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Getter
    @Description("Оферта для видеохостинга не прошла валидацию")
    public static class VideohostOfferFailed extends SiteProblemContent {
        private final String offerUrl;

        @JsonCreator
        public VideohostOfferFailed(@JsonProperty("offerUrl") String offerUrl) {
            super(SiteProblemTypeEnum.VIDEOHOST_OFFER_FAILED);
            this.offerUrl = offerUrl;
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }

    @Description("Требуется дополнительное бумажное соглашение для видеохостинга")
    public static class VideohostOfferNeedPaper extends SiteProblemContent {
        @JsonCreator
        public VideohostOfferNeedPaper() {
            super(SiteProblemTypeEnum.VIDEOHOST_OFFER_NEED_PAPER);
        }

        @Override
        public SiteProblemContent merge(SiteProblemContent that) {
            return this;
        }
    }
}
