package ru.yandex.webmaster3.core.codes;

import ru.yandex.webmaster3.core.util.enums.EnumResolver;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.Set;

/**
 * @author tsyplyaev
 */
public enum HttpCodeGroup implements ICodeGroup {
    UNKNOWN(-1),
    NEVER_DOWNLOADED(-2),

    HTTP_0_NOT_DOWNLOADED(0),

    HTTP_100_CONTINUE(100),
    HTTP_101_SWITCHING_PROTOCOLS(101),

    HTTP_GROUP_1XX(
            HTTP_100_CONTINUE,
            HTTP_101_SWITCHING_PROTOCOLS
    ),
    HTTP_GROUP_1XX_ALL(true, 100, 200),

    HTTP_200_OK(200),
    HTTP_201_CREATED(201),
    HTTP_202_ACCEPTED(202),
    HTTP_203_NON_AUTHORITATIVE_INFORMATION(203),
    HTTP_204_NO_CONTENT(204),
    HTTP_205_RESET_CONTENT(205),
    HTTP_206_PARTIAL_CONTENT(206),

    HTTP_300_MULTIPLE_CHOICES(300),
    HTTP_301_MOVED_PERMANENTLY(301),
    HTTP_302_FOUND(302),
    HTTP_303_SEE_OTHER(303),
    HTTP_304_NOT_MODIFIED(304),
    HTTP_305_USE_PROXY(305),
    HTTP_306_SWITCH_PROXY(306),
    HTTP_307_TEMPORARY_REDIRECT(307),
    HTTP_308_PERMANENT_REDIRECT(308),

    HTTP_400_BAD_REQUEST(400),
    HTTP_401_UNAUTHORIZED(401),
    HTTP_402_PAYMENT_REQUIRED(402),
    HTTP_403_FORBIDDEN(403),
    HTTP_404_NOT_FOUND(404),
    HTTP_405_METHOD_NOT_ALLOWED(405),
    HTTP_406_NOT_ACCEPTABLE(406),
    HTTP_407_PROXY_AUTHENTICATION_REQUIRED(407),
    HTTP_408_REQUEST_TIMEOUT(408),
    HTTP_409_CONFLICT(409),
    HTTP_410_GONE(410),
    HTTP_411_LENGTH_REQUIRED(411),
    HTTP_412_PRECONDITION_FAILED(412),
    HTTP_413_REQUEST_ENTITY_TOO_LARGE(413),
    HTTP_414_REQUEST_URI_TOO_LONG(414),
    HTTP_415_UNSUPPORTED_MEDIA_TYPE(415),
    HTTP_416_REQUESTED_RANGE_NOT_SATISFIABLE(416),
    HTTP_417_EXPECTATION_FAILED(417),
    HTTP_418_IM_A_TEAPOT(418),
    HTTP_419_AUTHENTICATION_TIMEOUT(419),
    HTTP_426_UPGRADE_REQUIRED(426),
    HTTP_428_PRECONDITION_REQUIRED(428),
    HTTP_429_TOO_MANY_REQUESTS(429),
    HTTP_431_REQUEST_HEADER_FIELDS_TOO_LARGE(431),

    HTTP_500_INTERNAL_SERVER_ERROR(500),
    HTTP_501_NOT_IMPLEMENTED(501),
    HTTP_502_BAD_GATEWAY(502),
    HTTP_503_SERVICE_UNAVAILABLE(503),
    HTTP_504_GATEWAY_TIMEOUT(504),
    HTTP_505_HTTP_VERSION_NOT_SUPPORTED(505),
    HTTP_506_VARIANT_ALSO_NEGOTIATES(506),
    HTTP_509_BANDWIDTH_LIMIT_EXCEEDED(509),
    HTTP_510_NOT_EXTENDED(510),
    HTTP_511_NETWORK_AUTHENTICATION_REQUIRED(511),


    HTTP_GROUP_2XX(
            HTTP_200_OK,
            HTTP_201_CREATED,
            HTTP_202_ACCEPTED,
            HTTP_203_NON_AUTHORITATIVE_INFORMATION,
            HTTP_204_NO_CONTENT,
            HTTP_205_RESET_CONTENT,
            HTTP_206_PARTIAL_CONTENT
    ),

    HTTP_GROUP_3XX(
            HTTP_301_MOVED_PERMANENTLY,
            HTTP_302_FOUND,
            HTTP_303_SEE_OTHER,
            HTTP_304_NOT_MODIFIED,
            HTTP_305_USE_PROXY,
            HTTP_306_SWITCH_PROXY,
            HTTP_307_TEMPORARY_REDIRECT,
            HTTP_308_PERMANENT_REDIRECT
    ),

    HTTP_GROUP_4XX(
            HTTP_400_BAD_REQUEST,
            HTTP_401_UNAUTHORIZED,
            HTTP_402_PAYMENT_REQUIRED,
            HTTP_403_FORBIDDEN,
            HTTP_404_NOT_FOUND,
            HTTP_405_METHOD_NOT_ALLOWED,
            HTTP_406_NOT_ACCEPTABLE,
            HTTP_407_PROXY_AUTHENTICATION_REQUIRED,
            HTTP_408_REQUEST_TIMEOUT,
            HTTP_409_CONFLICT,
            HTTP_410_GONE,
            HTTP_411_LENGTH_REQUIRED,
            HTTP_412_PRECONDITION_FAILED,
            HTTP_413_REQUEST_ENTITY_TOO_LARGE,
            HTTP_414_REQUEST_URI_TOO_LONG,
            HTTP_415_UNSUPPORTED_MEDIA_TYPE,
            HTTP_416_REQUESTED_RANGE_NOT_SATISFIABLE,
            HTTP_417_EXPECTATION_FAILED,
            HTTP_418_IM_A_TEAPOT,
            HTTP_419_AUTHENTICATION_TIMEOUT,
            HTTP_426_UPGRADE_REQUIRED,
            HTTP_428_PRECONDITION_REQUIRED,
            HTTP_429_TOO_MANY_REQUESTS,
            HTTP_431_REQUEST_HEADER_FIELDS_TOO_LARGE
    ),

    HTTP_GROUP_5XX(
            HTTP_500_INTERNAL_SERVER_ERROR,
            HTTP_501_NOT_IMPLEMENTED,
            HTTP_502_BAD_GATEWAY,
            HTTP_503_SERVICE_UNAVAILABLE,
            HTTP_504_GATEWAY_TIMEOUT,
            HTTP_505_HTTP_VERSION_NOT_SUPPORTED,
            HTTP_506_VARIANT_ALSO_NEGOTIATES,
            HTTP_509_BANDWIDTH_LIMIT_EXCEEDED,
            HTTP_510_NOT_EXTENDED
    ),

    HTTP_GROUP_1001(1001),
    HTTP_GROUP_1002(1002, 2015),
    HTTP_GROUP_1003(1003, 2021),
    HTTP_GROUP_1004(1004),
    HTTP_GROUP_1005(1005),
    HTTP_GROUP_1007(1007),
    HTTP_GROUP_1008(1008, 1009, 1010, 1013, 1014, 1015, 1016, 1017, 1018, 1019, 1020, 1024, 1027, 2020),
    HTTP_GROUP_1006(1006, 1010, 1029, 1030, 1031, 1032, 1033, 1034, 1035, 1036, 1037, 1041),
    HTTP_GROUP_1021(1021),
    HTTP_GROUP_1042(1042),
    HTTP_GROUP_1XXX_SPECIAL(
            HTTP_GROUP_1001,
            HTTP_GROUP_1002,
            HTTP_GROUP_1003,
            HTTP_GROUP_1005,
            HTTP_GROUP_1007,
            HTTP_GROUP_1008,
            HTTP_GROUP_1006,
            HTTP_GROUP_1021
    ),
    HTTP_GROUP_1XXX(true, 1000, 2000),

    HTTP_GROUP_2005(2005),
    HTTP_GROUP_2006(2006),
    HTTP_GROUP_2007(2007),
    HTTP_GROUP_2010(2010),
    HTTP_GROUP_2011(2011, 2012),
    HTTP_GROUP_2013(2013),
    HTTP_GROUP_2014(2014, 2024),
    HTTP_GROUP_2016(2016),
    HTTP_GROUP_2020(2020),
    HTTP_GROUP_2025(2025),

    HTTP_GROUP_2XXX_SPECIAL(
            HTTP_GROUP_2005,
            HTTP_GROUP_2006,
            HTTP_GROUP_2007,
            HTTP_GROUP_2010,
            HTTP_GROUP_2011,
            HTTP_GROUP_2013,
            HTTP_GROUP_2014,
            HTTP_GROUP_2016,
            HTTP_GROUP_2020,
            HTTP_GROUP_2025
    ),

    HTTP_GROUP_2XXX(true, 2000, 3000),
    HTTP_GROUP_3XXX(true, 3000, 4000);

    private static final Set<Integer> convertsTo200InDownloaded = new HashSet<>(Arrays.asList(2005, 2010, 2011, 2012, 2014, 2015, 2020, 2024, 2025));
    private final Set<Integer> codes = new LinkedHashSet<>();
    private final boolean searchable;

    HttpCodeGroup(Integer... codes) {
        this.codes.addAll(Arrays.asList(codes));
        this.searchable = true;
    }

    HttpCodeGroup(boolean flags, int from, int to) {
        for (int i = from; i < to; i++) {
            this.codes.add(i);
        }
        this.searchable = true;
    }

    HttpCodeGroup(HttpCodeGroup... groups) {
        for (HttpCodeGroup g : groups) {
            this.codes.addAll(g.codes);
        }
        this.searchable = false;
    }

    public static HttpCodeGroup get(Integer inputCode) {
        if (inputCode == null) {
            return UNKNOWN;
        }

        for (HttpCodeGroup rcg : HttpCodeGroup.values()) {
            if (rcg.searchable && rcg.codes.contains(inputCode)) {
                return rcg;
            }
        }

        return UNKNOWN;
    }

    public static HttpCodeGroup getForDownloaded(Integer inputCode) {
        if (inputCode == null) {
            return UNKNOWN;
        }

        if (convertsTo200InDownloaded.contains(inputCode)) {
            return HTTP_200_OK;
        }

        for (HttpCodeGroup rcg : HttpCodeGroup.values()) {
            if (rcg.searchable && rcg.codes.contains(inputCode)) {
                return rcg;
            }
        }

        return UNKNOWN;
    }

    @Override
    public boolean contains(Integer code) {
        return codes.contains(code);
    }

    public boolean contains(HttpCodeGroup codeGroup) {
        return codes.containsAll(codeGroup.codes);
    }

    public Set<Integer> getCodesInternal() {
        return Collections.unmodifiableSet(codes);
    }

    public int value() {
        return codes.iterator().next();
    }

    public static EnumResolver<HttpCodeGroup> R = EnumResolver.er(HttpCodeGroup.class);
}
