package ru.yandex.webmaster3.core.digest;

import org.apache.commons.lang3.tuple.Pair;

import org.joda.time.DateTime;
import org.springframework.scheduling.support.CronSequenceGenerator;
import ru.yandex.webmaster3.core.util.enums.IntEnum;
import ru.yandex.webmaster3.core.util.enums.IntEnumResolver;
import ru.yandex.webmaster3.core.util.TimeUtils;

import java.time.temporal.ChronoUnit;
import java.util.*;
import java.util.stream.Collectors;
import java.util.stream.Stream;

/**
 * TODO: упростить, посмотреть, что из этого нужно. Что точно НЕ нужно:
 * 1. Кастомные расписания для разных типов выгрузок (класс нужен как раз для того, чтобы иметь единое расписание)
 * 2. Принимать где-либо ChronoUnit и т.д. - то же самое, расписание единое, снаружи не нужно думать о хроноюнитах
 * Created by ifilippov5 on 04.09.17.
 */
public class DigestSchedulerUtil {
    private static final String DIGEST_SCHEDULE_CRON = "59 59 23 * * 7";
    private static final CronSequenceGenerator DIGEST_SCHEDULE_GENERATOR = new CronSequenceGenerator(DIGEST_SCHEDULE_CRON, TimeUtils.ZONE_MSK);

    private static final int MAX_DIGEST_DURATION_WEEKS = 3;

    public static DateTime getLastDigestBefore(DateTime beforeDate) {
        Date dateResult = DIGEST_SCHEDULE_GENERATOR.next(new Date(beforeDate.minusWeeks(1).getMillis()));
        return new DateTime(dateResult.getTime());
    }

    /**
     * Разбивает промежуток now-windowTimeSize*windowTimeUnit .. now по windowTimeUnit-ам
     */
    public static List<DateTime> getSlidingWindow(ChronoUnit windowTimeUnit, int windowSize, DateTime now) {
        DateTime startDate = roundDate(now, windowTimeUnit);

        return Stream.iterate(startDate, date -> date.minus(windowTimeUnit.getDuration().toMillis()))
                .limit(windowSize)
                .collect(Collectors.toList());
    }

    /**
     * Разбивает промежуток между двумя последними дайджестами по windowTimeUnit-ам
     */
    public static List<DateTime> getFixedWindow(DigestTaskCategory category, ChronoUnit windowTimeUnit, DateTime now) {
        return getFixedWindow(Scheduler.get(category), windowTimeUnit, now);
    }

    public static List<DateTime> getFixedWindow(String cron, ChronoUnit windowTimeUnit, DateTime now) {
        Pair<DateTime, DateTime> digest = getLastTwoDigests(cron, now);
        DateTime startDate = roundDate(digest.getRight(), windowTimeUnit);
        DateTime endDate = roundDate(digest.getLeft(), windowTimeUnit);

        return Stream.iterate(startDate, date -> date.minus(windowTimeUnit.getDuration().toMillis()))
                .limit((startDate.getMillis() - endDate.getMillis()) / windowTimeUnit.getDuration().toMillis())
                .collect(Collectors.toList());
    }

    /**
     * Возвращает даты двух последних дайджестов
     */
    public static Pair<DateTime, DateTime> getLastTwoDigests(DigestTaskCategory category, DateTime now) {
        return getLastTwoDigests(Scheduler.get(category), now);
    }

    /**
     * Возвращает даты двух последних дайджестов
     */
    public static Pair<DateTime, DateTime> getLastTwoDigests(String cron, DateTime now) {
        CronSequenceGenerator generator = new CronSequenceGenerator(cron);
        DateTime oldestDate = now.minusWeeks(MAX_DIGEST_DURATION_WEEKS * 2);
        Date currentExecutionDate = generator.next(oldestDate.toDate());
        Date last0 = (Date)currentExecutionDate.clone();
        Date last1 = (Date)currentExecutionDate.clone();
        Date today = now.toDate();
        while (!currentExecutionDate.after(today)) {
            last0 = (Date)last1.clone();
            last1 = (Date)currentExecutionDate.clone();
            currentExecutionDate = generator.next(currentExecutionDate);
        }
        return Pair.of(new DateTime(last0), new DateTime(last1));
    }

    /**
     * Округляет дату до timeUnit
     */
    public static DateTime roundDate(DateTime date, ChronoUnit timeUnit) {
        switch (timeUnit) {
            case WEEKS:
                return date.withDayOfWeek(1).withTimeAtStartOfDay();
            case DAYS:
                return date.withTimeAtStartOfDay();
            case HALF_DAYS:
                DateTime startOfDay = date.withTimeAtStartOfDay();
                DateTime midNight = startOfDay.plusHours(12);
                return date.getMillis() - startOfDay.getMillis() > ChronoUnit.HALF_DAYS.getDuration().toMillis() ? midNight : startOfDay;
            case HOURS:
                date = date.minusMinutes(date.getMinuteOfHour());
            case MINUTES:
                date = date.minusSeconds(date.getSecondOfMinute());
            case SECONDS:
                date = date.minusMillis(date.getMillisOfSecond());
                return date;
        }

        return null;
    }

    public static class Scheduler {
        private static final String DEFAULT_CRON = "0 0 0 * * 1"; // at 00:00 on Monday

        private static final Map<DigestTaskCategory, String> CRON_SCHEDULE = new HashMap<>();

        static {
            CRON_SCHEDULE.put(DigestTaskCategory.SITE_PROBLEMS, "0 0 0 * * 1");
        }

        public static String get(DigestTaskCategory category) {
            return CRON_SCHEDULE.getOrDefault(category, DEFAULT_CRON);
        }
    }

    public enum DigestTaskCategory implements IntEnum {
        SITE_PROBLEMS(0),;

        private final int value;

        DigestTaskCategory(int value) {
            this.value = value;
        }

        public int getValue() {
            return value;
        }

        @Override
        public int value() {
            return value;
        }

        public static final IntEnumResolver<DigestTaskCategory> R = IntEnumResolver.r(DigestTaskCategory.class);
    }
}
