package ru.yandex.webmaster3.core.domain;

import org.apache.commons.lang3.StringUtils;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.core.io.Resource;
import ru.yandex.webmaster3.core.data.HostDomainInfo;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.util.IdUtils;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * @author aherman
 */
public class HostDomainInfoService {
    private static final String GLOB = "*";
    private static final String WWW_PREFIX = "www.";
    private static final String FTP_PREFIX = "ftp.";

    public static final int HOSTNAME_IS_OWNER = 0;
    public static final int HOSTNAME_WITHOUT_OWNER = -1;

    private Resource hostOwnersData;
    private Map<String, Node> rulesMap;

    public static boolean isSubdomain(WebmasterHostId hostId) {
        String name = hostId.getPunycodeHostname();
        if (name.startsWith("www.")) {
            name = name.substring("www.".length());
        }
        return name.chars().filter(c -> c == '.').count() > 1L;
    }

    public void init() throws IOException {
        try (BufferedReader br = new BufferedReader(
                new InputStreamReader(hostOwnersData.getInputStream(), StandardCharsets.UTF_8))) {
            rulesMap = readRules(br);
        }
    }

    static Map<String, Node> readRules(BufferedReader br) throws IOException {
        String line;
        Map<String, Node> result = new HashMap<>();

        while ((line = br.readLine()) != null) {
            line = StringUtils.trimToEmpty(line);
            if (line.startsWith("//")) {
                continue;
            }
            line = IdUtils.IDN.toASCII(line);

            boolean exclusion = line.startsWith("!");
            boolean isGlobRule = false;
            if (exclusion) {
                // Ignore exclusion
                continue;
            }
            if (line.startsWith(GLOB)) {
                isGlobRule = true;
                line = line.substring(2);
            }
            result.put(line, new Node(isGlobRule));
        }
        return result;
    }

    public HostDomainInfo getDomainInfo(String punycodeDomainName) {
        int ownerStart = getOwnerStart(punycodeDomainName);
        String fullPrefix = StringUtils.substring(punycodeDomainName, 0, ownerStart);
        int prefixEnd = -1;
        if (fullPrefix.startsWith(WWW_PREFIX)) {
            prefixEnd = WWW_PREFIX.length() - 1;
        } else if (fullPrefix.startsWith(FTP_PREFIX)) {
            prefixEnd = FTP_PREFIX.length() - 1;
        }

        return new HostDomainInfo(punycodeDomainName, prefixEnd, ownerStart);
    }

    String getOwner(String fullDomain) {
        int ownerStart = getOwnerStart(fullDomain);
        if (ownerStart == HOSTNAME_WITHOUT_OWNER) {
            return null;
        } else if (ownerStart == HOSTNAME_IS_OWNER) {
            return fullDomain;
        }
        return fullDomain.substring(ownerStart);
    }

    int getOwnerStart(String fullDomain) {
        List<Integer> parentDomainStarts = new ArrayList<>();
        int dotPosition = fullDomain.indexOf('.');
        while (dotPosition > 0) {
            parentDomainStarts.add(dotPosition + 1);
            dotPosition = fullDomain.indexOf('.', dotPosition + 1);
        }
        if (parentDomainStarts.isEmpty()) {
            return HOSTNAME_WITHOUT_OWNER;
        }

        int candidateOwnerStart = Integer.MAX_VALUE;
        for (int i = parentDomainStarts.size() - 1; i >= 0; i--) {
            String parentDomain = fullDomain.substring(parentDomainStarts.get(i));
            Node node = rulesMap.get(parentDomain);
            if (node == null) {
                continue;
            }

            if (node.isGlobRule()) {
                if (i > 0) {
                    candidateOwnerStart = i - 1;
                }
                if (i == 0) {
                    return HOSTNAME_IS_OWNER;
                }
            } else {
                if (candidateOwnerStart > i) {
                    candidateOwnerStart = i;
                }
            }
        }
        Node node = rulesMap.get(fullDomain);
        if (node != null) {
            if (!node.isGlobRule()) {
                return HOSTNAME_IS_OWNER;
            }
        }
        if (candidateOwnerStart == Integer.MAX_VALUE) {
            // return TLD
            return parentDomainStarts.get(parentDomainStarts.size() - 1);
        }
        return parentDomainStarts.get(candidateOwnerStart);
    }

    @Required
    public void setHostOwnersData(Resource hostOwnersData) {
        this.hostOwnersData = hostOwnersData;
    }

    void setRulesMap(Map<String, Node> rulesMap) {
        this.rulesMap = rulesMap;
    }

    static class Node {
        private final boolean globRule;

        public Node(boolean hasGlobRule) {
            this.globRule = hasGlobRule;
        }

        public boolean isGlobRule() {
            return globRule;
        }
    }
}
