package ru.yandex.webmaster3.core.download;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.PipedInputStream;
import java.io.PipedOutputStream;
import java.io.Writer;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CountDownLatch;

import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.SequenceWriter;
import com.fasterxml.jackson.dataformat.csv.CsvMapper;
import com.opencsv.CSVWriter;
import com.opencsv.bean.BeanToCsv;
import com.opencsv.bean.ColumnPositionMappingStrategy;
import com.opencsv.bean.CsvBindByName;
import org.apache.commons.io.IOUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.webmaster3.core.xcelite.NotLeakingXcelite;
import ru.yandex.webmaster3.core.xcelite.sheet.XceliteSheet;
import ru.yandex.webmaster3.core.xcelite.writer.SheetWriter;

/**
 * Created by ifilippov5 on 31.01.17.
 */
public class ExportUtil {
    private static final Logger log = LoggerFactory.getLogger(ExportUtil.class);

    private static final CsvMapper CSV_MAPPER = new CsvMapper();

    static {
        CSV_MAPPER.setPropertyNamingStrategy(PropertyNamingStrategy.UPPER_CAMEL_CASE);
    }

    public static <T> byte[] exportToCsv(List<T> samples, Class<T> clazz) throws IOException {
        ByteArrayOutputStream out = new ByteArrayOutputStream();
        Writer writer = new OutputStreamWriter(out);
        CSVWriter csvWriter = new CSVWriter(writer);
        BeanToCsv<T> bc = new BeanToCsv<>();
        ColumnPositionMappingStrategy<T> mappingStrategy = new ColumnPositionMappingStrategy<>();
        mappingStrategy.setType(clazz);
        Field[] columns = clazz.getDeclaredFields();
        String[] names = Arrays.stream(columns)
                .filter(f -> f.isAnnotationPresent(CsvBindByName.class))
                .map(Field::getName)
                .toArray(String[]::new);
        mappingStrategy.setColumnMapping(names);
        bc.write(mappingStrategy, csvWriter, samples);
        if (samples.isEmpty()) {
            csvWriter.writeNext(names);
        }
        csvWriter.close();

        return out.toByteArray();
    }

    public static <T> byte[] exportToXlsx(List<T> samples, Class<T> clazz) throws IOException {
        NotLeakingXcelite xcelite = new NotLeakingXcelite();
        XceliteSheet sheet = xcelite.createSheet("sheet1");
        SheetWriter<T> writer = sheet.getBeanWriter(clazz);
        writer.write(samples);
        return xcelite.getBytes();
    }

    public static <T extends MdsSerializable> MdsWriter<T> createWriter(
            FileFormat format, Class<T> clazz) throws IOException {
        switch (format) {
            case CSV:
                return new CsvSpreadsheetWriter<T>(clazz);
            case CSV_GZ:
            case TSV_GZ:
                return new GzipWriter<T>();
            case EXCEL:
                return new XlsxSpreadsheetWriter<T>(clazz);
            default:
                throw new RuntimeException("Unsupported file format " + format);
        }
    }

    private static class GzipWriter<T> implements MdsWriter<T> {
        private final OutputStream out;
        private final InputStream in;

        GzipWriter() throws IOException {
            var pipedOutputStream = new PipedOutputStream();
            in = new PipedInputStream(pipedOutputStream);
            out = pipedOutputStream;
        }

        @Override
        public void write(T item) throws IOException {
            out.write(((GzipMdsSerializable) item).getValue());
        }

        @Override
        public InputStream inputStream() {
            return in;
        }

        @Override
        public void allDataUploaded() {
            try {
                out.flush();
                //По мотивам: WMCSUPPORT-3629
                //Дожидаемся когда все данные будут вычитаны
                while (in.available() > 0) {
                    Thread.sleep(100);
                }
                //Закрываем входящий поток, что бы принимающий поток не завис
                IOUtils.closeQuietly(out);
            } catch (IOException ioException) {
                log.error(ioException.getMessage(), ioException);
            } catch (InterruptedException e) {
                log.error(e.getMessage(), e);
            }
        }

        @Override
        public void close() throws IOException {

            IOUtils.closeQuietly(in);

        }
    }

    private static class CsvSpreadsheetWriter<T> implements MdsWriter<T> {
        private final SequenceWriter writer;
        private final OutputStream out;
        private final InputStream in;

        CsvSpreadsheetWriter(Class<T> clazz) throws IOException {
            var pipedOutputStream = new PipedOutputStream();
            in = new PipedInputStream(pipedOutputStream);
            out = pipedOutputStream;
            writer = CSV_MAPPER.writerFor(clazz)
                    .withSchema(CSV_MAPPER.schemaFor(clazz).withUseHeader(true))
                    .writeValues(pipedOutputStream);
        }

        @Override
        public void write(T item) throws IOException {
            writer.write(item);
        }

        @Override
        public InputStream inputStream() {
            return in;
        }

        @Override
        public void allDataUploaded() {
            try {
                out.flush();
                writer.flush();
                //По мотивам: WMCSUPPORT-3629
                //Дожидаемся когда все данные будут вычитаны
                while (in.available() > 0) {
                    Thread.sleep(100);
                }
                //Закрываем входящий поток, что бы принимающий поток не завис
                IOUtils.closeQuietly(out);
            } catch (IOException ioException) {
                log.error(ioException.getMessage(), ioException);
            } catch (InterruptedException e) {
                log.error(e.getMessage(), e);
            }
        }

        @Override
        public void close() throws IOException {
            writer.close();
            IOUtils.closeQuietly(in);
        }
    }

    /**
     * Xlsx это такой формат, что внутри него есть ссылки вперед, поэтому
     * его не получится стримить как CSV, построчно.
     * Вместо этого данные сначала пишутся во временный файл, кусочками,
     * то есть весь файл в памяти не хранится (этим занимается xcelite),
     * и затем уже все сразу заливается в OutputStream.
     */
    private static class XlsxSpreadsheetWriter<T> implements MdsWriter<T> {
        private final NotLeakingXcelite xcelite;
        private final SheetWriter<T> writer;
        private ByteArrayInputStream in;
        private CountDownLatch downLatch = new CountDownLatch(1);


        XlsxSpreadsheetWriter(Class<T> clazz) throws IOException {
            this.xcelite = new NotLeakingXcelite();
            XceliteSheet sheet = xcelite.createSheet("sheet1");
            this.writer = sheet.getBeanWriter(clazz);
        }

        @Override
        public void write(T item) throws IOException {
            writer.write(Collections.singleton(item));
        }

        @Override
        public InputStream inputStream() {
            try {
                downLatch.await();
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            if (in == null) {
                throw new RuntimeException("in is null, something wrong in data provider");
            }
            return in;
        }

        @Override
        public void allDataUploaded() {
            try {
                in = new ByteArrayInputStream(xcelite.getBytes());
                downLatch.countDown();
            } catch (IOException ioException) {

            }
        }

        @Override
        public void close() throws IOException {
            downLatch.countDown();
            in.close();
        }
    }
}
