package ru.yandex.webmaster3.core.events2;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.fasterxml.jackson.annotation.JsonTypeInfo;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationContext;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.annotation.JsonTypeIdResolver;
import com.fasterxml.jackson.databind.annotation.JsonTypeResolver;
import com.fasterxml.jackson.databind.deser.std.StdDeserializer;
import com.fasterxml.jackson.databind.node.TreeTraversingParser;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import com.fasterxml.jackson.module.paramnames.ParameterNamesModule;
import com.google.common.collect.ImmutableMap;
import org.joda.time.Instant;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;
import ru.yandex.webmaster3.core.util.IdUtils;
import ru.yandex.webmaster3.core.util.json.polymorphic.DefaultValuedTypeResolverBuilder;
import ru.yandex.webmaster3.core.util.json.polymorphic.DiscriminatorBasedTypeResolver;

import java.io.IOException;
import java.util.UUID;

/**
 * Для сохранения на диск/в БД используем немного другой формат сериализации:
 * тип event'а выводится не внутри поля data, а снаружи (то есть при сериализации отдельно HostEventData тип вообще теряется).
 *
 * Зачем так сделано:
 * В кликхаусе тип в любом случае лежит отдельным полем, чтобы по нему можно было фильтровать и пр.
 * - и хотелось избавиться от дублирования типа внутри json'а
 * При сериализации всего HostEvent (например на диск) так делается уже для единообразия.
 * @author avhaliullin
 */
public class HostEventJsonUtils {
    private static final ObjectMapper OM = new ObjectMapper();

    static {
        OM.disable(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES);
        OM.addMixIn(HostEventData.class, HostEventDataHideType.class);
        OM.addMixIn(HostEvent.class, HostEventMixIn.class);
        OM.registerModule(new JodaModule());
        OM.registerModule(new WebmasterJsonModule(false));
        OM.registerModule(new ParameterNamesModule());
    }

    public static String serializeEvent(HostEvent event) {
        try {
            return OM.writeValueAsString(event);
        } catch (JsonProcessingException e) {
            throw new WebmasterException("Failed to serialize host event of type " + event.getType(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(HostEventJsonUtils.class, null), e);
        }
    }

    public static HostEvent deserializeEvent(String data) {
        try {
            return OM.readValue(data, HostEvent.class);
        } catch (IOException e) {
            throw new WebmasterException("Failed to deserialize host event from string " + data,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(HostEventJsonUtils.class, null), e);
        }
    }

    public static String serialize(HostEventData data) {
        try {
            return OM.writeValueAsString(data);
        } catch (JsonProcessingException e) {
            throw new WebmasterException("Failed to serialize host event data of type " + data.getType(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(HostEventJsonUtils.class, null), e);
        }
    }

    public static HostEventData deserialize(HostEventType type, String data) {
        try {
            return deserialize(type, OM.getFactory().createParser(data));
        } catch (IOException e) {
            throw new WebmasterException("Failed to deserialize host event data of type " + type + " from string " + data,
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(HostEventJsonUtils.class, null), e);
        }
    }

    private static HostEventData deserialize(HostEventType type, JsonNode node) {
        try {
            return deserialize(type, new TreeTraversingParser(node));
        } catch (IOException e) {
            throw new WebmasterException("Failed to deserialize host event data of type " + type + " from node " + node.toString(),
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(HostEventJsonUtils.class, null), e);
        }
    }

    private static HostEventData deserialize(HostEventType type, JsonParser parser) throws IOException {
        Class<? extends HostEventData> dataClass = type.getDataClass();
        return OM.readValue(parser, dataClass);

    }

    public interface HostEventMixIn {
        @JsonTypeInfo(use = JsonTypeInfo.Id.CUSTOM,
                include = JsonTypeInfo.As.EXTERNAL_PROPERTY,
                property = "type")
        @JsonTypeIdResolver(DiscriminatorBasedTypeResolver.class)
        HostEventData getData();
    }

    public interface HostEventDataHideType {
        @JsonIgnore
        HostEventType getType();
    }
}
