package ru.yandex.webmaster3.core.events2.client;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import org.apache.commons.io.IOUtils;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Required;
import org.springframework.web.util.UriComponentsBuilder;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.events2.HostEvent;
import ru.yandex.webmaster3.core.http.HttpConstants;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.http.WebmasterJsonModule;

import java.io.IOException;
import java.net.URI;

/**
 * @author avhaliullin
 */
public class RemoteHostEventLogClient extends AbstractHostEventLogClient {
    private static final ObjectMapper OM = new ObjectMapper()
            .registerModules(new JodaModule(), new WebmasterJsonModule(false));

    private static final int SOCKET_TIMEOUT_MS = 2_000;

    private String receiverUrl;
    private String applicationName;
    private String applicationVersion;
    private CloseableHttpClient httpClient;

    public RemoteHostEventLogClient() {
        super(5);
    }

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(HttpConstants.DEFAULT_CONNECTION_REQUEST_TIMEOUT)
                .setConnectTimeout(HttpConstants.DEFAULT_CONNECT_TIMEOUT)
                .setSocketTimeout(SOCKET_TIMEOUT_MS)
                .build();

        httpClient = HttpClients.custom()
                .setMaxConnTotal(20)
                .setMaxConnPerRoute(20)
                .setDefaultRequestConfig(requestConfig)
                .build();
    }

    public void destroy() {
        IOUtils.closeQuietly(httpClient);
    }

    @Override
    protected void doLog(HostEvent event) throws IOException {
        String reqId = UUIDs.timeBased().toString();
        URI requestUri = UriComponentsBuilder.fromUriString(receiverUrl)
                .path("/host/addEvent.json")
                .queryParam("applicationName", applicationName)
                .queryParam("applicationVersion", applicationVersion)
                .queryParam("requestId", reqId)
                .build()
                .toUri();
        HttpPost post = new HttpPost(requestUri);

        String dataString = "{\"event\":" + OM.writeValueAsString(event) + "}";
        post.setEntity(new StringEntity(dataString, ContentType.APPLICATION_JSON));
        try (CloseableHttpResponse response = httpClient.execute(post)) {
            if (response.getStatusLine().getStatusCode() != 200) {
                throw new WebmasterException("Got code " + response.getStatusLine().getStatusCode() + " on request " + reqId,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null));
            }
            JsonNode responseNode = OM.readTree(response.getEntity().getContent());
            if (!responseNode.has("status") || !responseNode.get("status").asText().equals("SUCCESS")) {
                throw new WebmasterException("Status was not success on request " + reqId + ": " + responseNode.toString(),
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), null));
            }
        }
    }

    @Required
    public void setReceiverUrl(String receiverUrl) {
        this.receiverUrl = receiverUrl;
    }

    @Required
    public void setApplicationName(String applicationName) {
        this.applicationName = applicationName;
    }

    @Required
    public void setApplicationVersion(String applicationVersion) {
        this.applicationVersion = applicationVersion;
    }
}
