package ru.yandex.webmaster3.core.feeds.mbi;

import java.io.IOException;
import java.net.URI;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.TimeUnit;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.io.IOUtils;
import org.apache.http.HttpEntity;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.metrics.externals.ExternalDependencyMethod;
import ru.yandex.webmaster3.core.util.JavaMethodWitness;
import ru.yandex.wmtools.common.sita.UserAgentEnum;

/**
 * @author kravchenko99
 * @date 7/28/22
 */
@Service
@Slf4j
public class AboService extends AbstractExternalAPIService {

    private static final String OFF_BY_PARTNER = "/fullModeration";
    private CloseableHttpClient httpClient;
    @Value("${webmaster3.storage.abo.address}")
    private URI mbiHost;

    public void init() {
        RequestConfig requestConfig = RequestConfig.custom()
                .setConnectionRequestTimeout(1)
                .setConnectTimeout(100)
                .setSocketTimeout(5 * 60_000)
                .build();

        httpClient = HttpClients.custom()
                .setUserAgent(UserAgentEnum.WEBMASTER.getValue())
                .setMaxConnPerRoute(150)
                .setMaxConnTotal(150)
                .setConnectionTimeToLive(5, TimeUnit.SECONDS)
                .setDefaultRequestConfig(requestConfig)
                .disableCookieManagement()
                .build();
    }

    @ExternalDependencyMethod("start-full-scc")
    public void startFullScc(long businessId) {
        trackQuery(new JavaMethodWitness() {
        }, ALL_ERRORS_INTERNAL, () -> {
            String uri = mbiHost.toString() + OFF_BY_PARTNER +
                    "?businessId=" + businessId;
            HttpPost httpReq = new HttpPost(uri);
            log.info("Abo start full scc: {}", businessId);

            try (CloseableHttpResponse response = httpClient.execute(httpReq)) {
//                log.info("x-market-req-id: {} for shopId {}", response.getFirstHeader("x-market-req-id"), shopId);

                HttpEntity responseEntity = response.getEntity();
                int httpCode = response.getStatusLine().getStatusCode();
                if (httpCode != 200) {
                    if (responseEntity != null) {
                        log.error("ABO responded httpCode - {} , body - {}", httpCode,
                                IOUtils.toString(responseEntity.getContent(), StandardCharsets.UTF_8));
                    } else {
                        log.error("ABO responded httpCode - {}", httpCode);
                    }
                    throw new IllegalStateException("MBI return httpCode " + httpCode);
                }
                return null;
            } catch (IOException e) {
                throw new WebmasterException("Abo request to start full scc failed for - " + businessId,
                        new WebmasterErrorResponse.InternalUnknownErrorResponse(getClass(), "Abo req fail"), e);
            }
        });
    }

}
