package ru.yandex.webmaster3.core.host.service;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.ObjectUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import ru.yandex.webmaster3.core.data.WebmasterHostId;
import ru.yandex.webmaster3.core.domain.PublicSuffixListCacheService;
import ru.yandex.webmaster3.core.metrics.externals.AbstractExternalAPIService;
import ru.yandex.webmaster3.core.util.IdUtils;

import java.io.IOException;
import java.util.*;
import java.util.stream.Collectors;

/**
 * @author avhaliullin
 */
@Slf4j
@Service("hostOwnerService")
public class HostOwnerService extends AbstractExternalAPIService {
    private HostCanonizer.HostCanonizerWrapper hostCanonizer;
    private final PublicSuffixListCacheService publicSuffixListCacheService;

    @Autowired
    public HostOwnerService(PublicSuffixListCacheService publicSuffixListCacheService) {
        this(new HostCanonizer.HostCanonizerWrapper(), publicSuffixListCacheService);
    }

    public HostOwnerService(HostCanonizer.HostCanonizerWrapper hostCanonizer, PublicSuffixListCacheService publicSuffixListCacheService) {
        this.hostCanonizer = hostCanonizer;
        this.publicSuffixListCacheService = publicSuffixListCacheService;
    }

    public boolean isSameOwner(WebmasterHostId hostId1, WebmasterHostId hostId2) {
        Map<WebmasterHostId, WebmasterHostId> hostId2Owner =
                mapHostsToOwners(Arrays.asList(hostId1, hostId2));

        return hostId2Owner.get(hostId1).equals(hostId2Owner.get(hostId2));
    }

    public Map<WebmasterHostId, WebmasterHostId> mapHostsToOwners(List<WebmasterHostId> hostIds) {
        Map<WebmasterHostId, WebmasterHostId> result = new HashMap<>();
        for (WebmasterHostId hostId : hostIds) {
            String owner = hostCanonizer.getHostOwner(IdUtils.hostIdToUrl(hostId));
            result.put(hostId, owner == null ? hostId : IdUtils.urlToHostId(owner));
        }
        return result;
    }

    public Map<WebmasterHostId, WebmasterHostId> mapHostsToMascotOwners(List<WebmasterHostId> hostIds) {
        Map<WebmasterHostId, WebmasterHostId> result = new HashMap<>();
        for (WebmasterHostId hostId : hostIds) {
            String owner = hostCanonizer.getMascotHostOwner(IdUtils.hostIdToUrl(hostId));
            result.put(hostId, owner == null ? hostId : IdUtils.urlToHostId(owner));
        }
        return result;
    }

    public Map<String, String> mapDomainsToOwners(Collection<String> domains) {
        List<WebmasterHostId> hostIds = domains.stream().map(IdUtils::urlToHostId).collect(Collectors.toList());

        List<WebmasterHostId> batchHostIds = new ArrayList<>();
        Map<WebmasterHostId, WebmasterHostId> host2Owner = new HashMap<>();
        for (WebmasterHostId hostId : hostIds) {
            batchHostIds.add(hostId);
            if (batchHostIds.size() == 1024) {
                host2Owner.putAll(mapHostsToOwners(batchHostIds));
                batchHostIds.clear();
            }
        }
        if (batchHostIds.size() > 0) {
            host2Owner.putAll(mapHostsToOwners(batchHostIds));
        }

        Map<String, String> domain2Owner = new HashMap<>();
        for (var entry : host2Owner.entrySet()) {
            WebmasterHostId hostIdKey = entry.getKey();
            WebmasterHostId hostIdValue = entry.getValue();
            domain2Owner.put(hostIdKey.getASCIIHostname(), hostIdValue.getASCIIHostname());
        }

        return domain2Owner;
    }

    @NotNull
    public WebmasterHostId getHostOwner(WebmasterHostId hostId) {
        return mapHostsToOwners(Collections.singletonList(hostId)).get(hostId);
    }

    public WebmasterHostId getMascotHostOwner(WebmasterHostId hostId) {
        return mapHostsToMascotOwners(Collections.singletonList(hostId)).get(hostId);
    }

    public String getHostOwner(@NotNull String host) {
        return ObjectUtils.firstNonNull(hostCanonizer.getHostOwner(host.toLowerCase()), host);
    }

    public String getMascotHostOwner(@NotNull String host) {
        return ObjectUtils.firstNonNull(hostCanonizer.getMascotHostOwner(host.toLowerCase()), host);
    }

    public String getLongestOwner(WebmasterHostId hostId) {
        return getLongestOwner(hostId, getHostOwner(hostId));
    }

    public Map<WebmasterHostId, String> getLongestOwners(List<WebmasterHostId> hostIds) {
        Map<WebmasterHostId, String> result = new HashMap<>();
        Map<WebmasterHostId, WebmasterHostId> owners = mapHostsToOwners(hostIds);

        owners.forEach((hostId, owner) -> {
            String longestOwner = getLongestOwner(hostId, owner);
            result.put(hostId, longestOwner);
        });
        return result;
    }

    public String getLongestOwner(WebmasterHostId hostId, WebmasterHostId owner) {
        String longestOwner = owner.getPunycodeHostname();
        try {
            //результат может начинаться с www
            String registrableDomain = publicSuffixListCacheService.getOwner(hostId.getPunycodeHostname());
            if (registrableDomain != null && registrableDomain.length() > longestOwner.length()) {
                longestOwner = registrableDomain;
            }
        } catch (IOException e) {
            log.error("Unable to find owner in public suffix list", e);
        }
        return longestOwner;
    }
}
