package ru.yandex.webmaster3.core.host.verification;

import java.util.UUID;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonIgnore;
import lombok.ToString;
import lombok.With;
import org.jetbrains.annotations.Nullable;
import org.joda.time.Duration;
import org.joda.time.Instant;

import ru.yandex.webmaster3.core.data.WebmasterHostId;

/**
 * @author avhaliullin
 */
@With
@ToString
public class UserHostVerificationInfo {
    private final UUID recordId;
    private final long userId;
    private final WebmasterHostId hostId;
    private final long verificationUin;
    private final VerificationType verificationType;
    private final VerificationStatus verificationStatus;
    private final int failedAttempts;
    private final VerificationFailInfo verificationFailInfo;
    private final Instant nextAttempt;
    private final Instant lastAttempt;
    private final boolean addedToList;
    private final VerificationCausedBy verificationCausedBy;
    private Long initiatorId;

    public UserHostVerificationInfo(UUID recordId, long userId, WebmasterHostId hostId, long verificationUin,
                                    VerificationType verificationType, VerificationStatus verificationStatus,
                                    int failedAttempts, VerificationFailInfo verificationFailInfo, Instant nextAttempt,
                                    Instant lastAttempt, boolean addedToList, VerificationCausedBy verificationCausedBy) {
        this.recordId = recordId;
        this.userId = userId;
        this.hostId = hostId;
        this.verificationUin = verificationUin;
        this.verificationType = verificationType;
        this.verificationStatus = verificationStatus;
        this.failedAttempts = failedAttempts;
        this.verificationFailInfo = verificationFailInfo;
        this.nextAttempt = nextAttempt;
        this.lastAttempt = lastAttempt;
        this.addedToList = addedToList;
        this.verificationCausedBy = verificationCausedBy;
    }

    public UserHostVerificationInfo(UUID recordId, long userId, WebmasterHostId hostId, long verificationUin,
                                    VerificationType verificationType, VerificationStatus verificationStatus,
                                    int failedAttempts, VerificationFailInfo verificationFailInfo, Instant nextAttempt,
                                    Instant lastAttempt, boolean addedToList, VerificationCausedBy verificationCausedBy, Long initiatorId) {
        this(recordId, userId, hostId, verificationUin, verificationType, verificationStatus, failedAttempts, verificationFailInfo,
                nextAttempt, lastAttempt, addedToList, verificationCausedBy);
        this.initiatorId = initiatorId;
    }

    public UUID getRecordId() {
        return recordId;
    }

    @JsonIgnore
    public Instant getCreateDate() {
        return new Instant(UUIDs.unixTimestamp(recordId));
    }

    public long getUserId() {
        return userId;
    }

    public WebmasterHostId getHostId() {
        return hostId;
    }

    public long getVerificationUin() {
        return verificationUin;
    }

    @Nullable
    public VerificationType getVerificationType() {
        return verificationType;
    }

    @Nullable
    public VerificationStatus getVerificationStatus() {
        return verificationStatus;
    }

    public int getFailedAttempts() {
        return failedAttempts;
    }

    public VerificationFailInfo getVerificationFailInfo() {
        return verificationFailInfo;
    }

    public Instant getNextAttempt() {
        return nextAttempt;
    }

    public Instant getLastAttempt() {
        return lastAttempt;
    }

    public boolean isAddedToList() {
        return addedToList;
    }

    @Nullable
    public VerificationCausedBy getVerificationCausedBy() {
        return verificationCausedBy;
    }

    public Long getInitiatorId() {
        return initiatorId;
    }

    public UserHostVerificationInfo copyWithIntermediateFail(UUID recordId, VerificationFailInfo failInfo, Duration retryAfter) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.IN_PROGRESS, failedAttempts + 1, failInfo, Instant.now().plus(retryAfter),
                Instant.now(), true, verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyWithTotallyFail(UUID recordId, VerificationFailInfo failInfo) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.VERIFICATION_FAILED, failedAttempts + 1, failInfo, null, Instant.now(), true,
                verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyAsVerified(UUID recordId) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.VERIFIED, failedAttempts, null, null, Instant.now(), true,
                verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyWithInternalError(UUID recordId) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.INTERNAL_ERROR, failedAttempts, null, null, Instant.now(), true,
                verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyAsVerificationStartRecord(UUID recordId, VerificationType verificationType, VerificationCausedBy verificationCausedBy) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.IN_PROGRESS, 0, null, Instant.now(), null, true,
                verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyDelegated(UUID recordId) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, VerificationType.DELEGATED,
                VerificationStatus.VERIFIED, 0, null, null, Instant.now(), true,
                verificationCausedBy, initiatorId);
    }

    public UserHostVerificationInfo copyCancelled(UUID recordId) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                null, 0, null, null, null, true,
                verificationCausedBy, initiatorId);
    }

    public static UserHostVerificationInfo createVerificationStartRecord(
            UUID recordId, long userId, WebmasterHostId hostId, VerificationType verificationType,
            long verificationUin, VerificationCausedBy verificationCausedBy) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.IN_PROGRESS, 0, null, Instant.now(), null, true,
                verificationCausedBy);
    }

    public static UserHostVerificationInfo createVerificationStartRecord(
            UUID recordId, long userId, WebmasterHostId hostId, VerificationType verificationType,
            long verificationUin, VerificationCausedBy verificationCausedBy, long initiatorId) {
        return new UserHostVerificationInfo(recordId, userId, hostId, verificationUin, verificationType,
                VerificationStatus.IN_PROGRESS, 0, null, Instant.now(), null, true,
                verificationCausedBy, initiatorId);
    }

    public static UserHostVerificationInfo createUinInitRecord(UUID recordId, long userId, WebmasterHostId hostId,
                                                               long uin, VerificationType verificationType,
                                                               boolean addedToList, VerificationCausedBy verificationCausedBy) {
        return new UserHostVerificationInfo(recordId, userId, hostId, uin, verificationType, null, 0, null, null, null, addedToList,
                verificationCausedBy);
    }
}
