package ru.yandex.webmaster3.core.host.verification;

import com.fasterxml.jackson.annotation.JsonProperty;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.data.HttpResponsePart;
import ru.yandex.webmaster3.core.util.ZoraResponseDocumentUtil;
import ru.yandex.webmaster3.core.zora.data.response.ZoraUrlFetchResponse;
import ru.yandex.wmtools.common.error.InternalException;

import java.io.IOException;
import java.util.List;

/**
 * @author avhaliullin
 */
public class VerificationHttpResponseInfo {
    private static final Logger log = LoggerFactory.getLogger(VerificationHttpResponseInfo.class);

    private static final int MAX_RESPONSE_LENGTH_SYMBOLS = 10000;

    private final HttpResponsePart httpResponsePart;
    private final List<String> redirectChain;
    private final boolean responseTruncated;

    public VerificationHttpResponseInfo(
            @JsonProperty("httpResponsePart") HttpResponsePart httpResponsePart,
            @JsonProperty("redirectChain") List<String> redirectChain,
            @JsonProperty("responseTruncated") boolean responseTruncated
    ) {
        this.httpResponsePart = httpResponsePart;
        this.redirectChain = redirectChain;
        this.responseTruncated = responseTruncated;
    }

    public HttpResponsePart getHttpResponsePart() {
        return httpResponsePart;
    }

    public List<String> getRedirectChain() {
        return redirectChain;
    }

    public boolean isResponseTruncated() {
        return responseTruncated;
    }

    public static VerificationHttpResponseInfo createFromZoraResponse(ZoraUrlFetchResponse zoraResponse) {
        String documentContent = null;
        boolean responseTruncated = false;
        if (zoraResponse.hasDocument()) {
            try {
                documentContent = ZoraResponseDocumentUtil.getResponseString(zoraResponse);

                if (documentContent.length() > MAX_RESPONSE_LENGTH_SYMBOLS) {
                    documentContent = documentContent.substring(0, MAX_RESPONSE_LENGTH_SYMBOLS);
                    responseTruncated = true;
                }
            } catch (IOException | InternalException e) {
                log.error("Failed to get document content", e);
            }
        }
        HttpResponsePart responsePart = HttpResponsePart.createFromZoraResponse(zoraResponse, documentContent);
        return new VerificationHttpResponseInfo(
                responsePart,
                ZoraResponseDocumentUtil.getRedirectsList(zoraResponse),
                responseTruncated
        );
    }
}
