package ru.yandex.webmaster3.core.host.verification;

import java.util.Collections;
import java.util.EnumSet;
import java.util.Optional;
import java.util.Set;

import org.joda.time.Duration;

import ru.yandex.webmaster3.core.util.enums.IntEnum;
import ru.yandex.webmaster3.core.util.enums.IntEnumResolver;

/**
 * @author avhaliullin
 */
public enum VerificationType implements IntEnum {
    // canBeInherited -- можно ли раздавать права , мб стоит переименовать в canBeDelegated , displayable -- какие подтверждения пользователи видят снаружи
    UNKNOWN(0, Duration.standardDays(1), false, false, RetryingPolicy.never(), false, false),
    SELF(1, Duration.standardDays(7), false, false, RetryingPolicy.never(), false, false),
    HTML_FILE(3, Duration.standardDays(180), true, true, RetryingPolicy.never(), true, true),
    META_TAG(2, Duration.standardDays(180), true, true, RetryingPolicy.never(), true, true),
    DNS(4, Duration.standardDays(180), true, true, RetryingPolicy.linearBackoff(Duration.standardMinutes(1), 2), true, true),
    WHOIS(5, Duration.standardDays(180), false, true, RetryingPolicy.never(), true, true),
    DELEGATED(6, Duration.standardDays(7), false, true, RetryingPolicy.never(), false, false),
    // WMC-7524: ПДД закрыли, проверка оставлена чтобы не ломать перепроверку. Пользователям недоступна
    PDD(7, Duration.standardDays(180), false, true, RetryingPolicy.never(), false, true, EnumSet.noneOf(VerificationTypeScope.class)),
    AUTO(8, Duration.standardDays(180), false, true, RetryingPolicy.never(), false, true),
    IDM(9, Duration.standardDays(30), false, false, RetryingPolicy.never(), false, false),
    // WMC-6592
    PDD_EMU(11, Duration.standardDays(180), true, true,
            RetryingPolicy.never(), true, true, EnumSet.of(VerificationTypeScope.INTERNAL)),
    // WMC-8279
    FAKE(12, Duration.standardDays(180), false, false, RetryingPolicy.never(), false, false,EnumSet.of(VerificationTypeScope.INTERNAL)),
    ;

    private final int value;
    private final Duration reverifyPeriod;
    private final boolean canBeInherited;
    private final boolean displayable;
    private final RetryingPolicy retryingPolicy;
    private final boolean explicit;
    private final boolean canBeDelegated;
    private final Set<VerificationTypeScope> scopes;

    VerificationType(int value, Duration reverifyPeriod, boolean canBeInherited, boolean displayable,
                     RetryingPolicy retryingPolicy, boolean explicit, boolean canBeDelegated) {
        this.value = value;
        this.reverifyPeriod = reverifyPeriod;
        this.canBeInherited = canBeInherited;
        this.displayable = displayable;
        this.retryingPolicy = retryingPolicy;
        this.explicit = explicit;
        this.scopes = VerificationTypeScope.ALL;
        this.canBeDelegated = canBeDelegated;

    }

    VerificationType(int value, Duration reverifyPeriod, boolean canBeInherited, boolean displayable,
                     RetryingPolicy retryingPolicy, boolean explicit,
                     boolean canBeDelegated, EnumSet<VerificationTypeScope> scopes) {
        this.value = value;
        this.reverifyPeriod = reverifyPeriod;
        this.canBeInherited = canBeInherited;
        this.displayable = displayable;
        this.retryingPolicy = retryingPolicy;
        this.explicit = explicit;
        this.canBeDelegated = canBeDelegated;
        this.scopes = scopes;
    }

    public Duration getReverifyPeriod() {
        return reverifyPeriod;
    }

    public boolean isCanBeInherited() {
        return canBeInherited;
    }

    public boolean isCanBeDelegated() {
        return canBeDelegated;
    }

    public boolean isDisplayable() {
        return displayable;
    }

    public boolean isExplicit() {
        return explicit;
    }

    public Optional<Duration> shouldRetry(int failedAttempts) {
        return retryingPolicy.shouldRetry(failedAttempts);
    }

    public boolean visibleIn(VerificationTypeScope scope) {
        return scopes.contains(scope);
    }

    @Override
    public int value() {
        return value;
    }

    public static IntEnumResolver<VerificationType> R = IntEnumResolver.r(VerificationType.class);

    public static final Set<VerificationType> AUTO_VERIFIERS = Collections.unmodifiableSet(
            EnumSet.of(DELEGATED, AUTO, IDM)
    );


    /* Utilities */
    private interface RetryingPolicy {
        Optional<Duration> shouldRetry(int failedAttempts);

        static RetryingPolicy never() {
            return failedAttempts -> Optional.empty();
        }

        static RetryingPolicy linearBackoff(Duration resultingDuration, int attempts) {
            int maxAttempts = Math.max(2, attempts);
            Duration step = resultingDuration.multipliedBy(2).dividedBy((maxAttempts - 1) * maxAttempts);
            return failedAttempts -> {
                if (failedAttempts >= maxAttempts) {
                    return Optional.empty();
                } else {
                    return Optional.of(step.multipliedBy(failedAttempts));
                }
            };
        }
    }
}
