package ru.yandex.webmaster3.core.hoststat;

import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonCreator;
import com.fasterxml.jackson.annotation.JsonProperty;
import lombok.AllArgsConstructor;
import lombok.Value;
import org.joda.time.DateTime;

import ru.yandex.autodoc.common.doc.annotation.Description;

/**
 * Created by Oleg Bazdyrev on 16/03/2021.
 */
@Value
@AllArgsConstructor(onConstructor_ = @JsonCreator)
public final class HostStatisticsInfo {

    private static final HostStatisticsInfo DEFAULT = new HostStatisticsInfo(1L, 0L, 0L, 0L, 0L, Collections.emptyList(), Collections.emptyList());

    @Description("Сколько всего хороших страниц в поиске")
    long goodDocsOnSearch;
    @Description("Количество уникальных тайтлов")
    long uniqueTitles;
    @Description("Количество уникальных дескрипшенов")
    long uniqueDescriptions;
    @Description("Количество страниц с дублирующимся тайтлами")
    long duplicateTitles;
    @Description("Количество страниц с дублирующимся дескрипшенами")
    long duplicateDescriptions;
    @Description("Доминирующие тайтлы")
    List<ContentAttrSampleInfo> prevailingTitles;
    @Description("Доминирующие дескрипшены")
    List<ContentAttrSampleInfo> prevailingDescriptions;

    public static HostStatisticsInfo fromHostStatistics(HostStatistics s) {
        if (s == null) {
            return DEFAULT;
        }
        return new HostStatisticsInfo(
                s.getGoodDocsOnSearch(),
                s.getUniqueTitles(),
                s.getUniqueDescriptions(),
                s.getDuplicateTitles(),
                s.getDuplicateDescriptions(),
                s.getPrevailingTitles() == null ? null : s.getPrevailingTitles().stream()
                        .map(ContentAttrSampleInfo::fromContentAttrSample).collect(Collectors.toList()),
                s.getPrevailingDescriptions() == null ? null : s.getPrevailingDescriptions().stream()
                        .map(ContentAttrSampleInfo::fromContentAttrSample).collect(Collectors.toList())
        );
    }

    @Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static class PageSampleInfo {
        @JsonProperty("path")
        String path;
        @JsonProperty("lastAccess")
        DateTime lastAccess;
        @JsonProperty("mainUrl")
        String mainUrl;

        public PageSampleInfo(String path, DateTime lastAccess) {
            this(path, lastAccess, null);
        }

        private static PageSampleInfo fromPageSample(HostStatistics.PageSample s) {
            if (s == null) {
                return null;
            }
            return new PageSampleInfo(s.getPath(), s.getLastAccess() == null ? null : new DateTime(s.getLastAccess() * 1000L), s.getMainUrl());
        }
    }

    @Value
    @AllArgsConstructor(onConstructor_ = @JsonCreator)
    public static class ContentAttrSampleInfo {
        @JsonProperty("value")
        String value;
        @JsonProperty("count")
        Long count;
        @JsonProperty("pageSamples")
        List<PageSampleInfo> pageSamples;

        private static ContentAttrSampleInfo fromContentAttrSample(HostStatistics.ContentAttrSample s) {
            if (s == null) {
                return null;
            }
            return new ContentAttrSampleInfo(s.getValue(), s.getCount(),
                    s.getPageSamples().stream().map(PageSampleInfo::fromPageSample).collect(Collectors.toList()));
        }
    }
}
