package ru.yandex.webmaster3.core.http;

import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.springframework.beans.factory.annotation.Required;
import ru.yandex.autodoc.common.doc.view.Markup;
import ru.yandex.autodoc.common.doc.view.renderers.HtmlRenderer;
import ru.yandex.webmaster3.core.http.autodoc.ActionDocumentationProvider;
import ru.yandex.webmaster3.core.http.internal.ActionParameters;
import ru.yandex.webmaster3.core.http.internal.ActionReflectionUtils;

import javax.servlet.ServletException;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.List;

/**
 * @author aherman
 */
public class ActionInfoHandler extends AbstractHandler {
    private static final HtmlRenderer HTML_MARKUP_RENDERER = HtmlRenderer.INSTANCE;
    private static final String EXTENSION = ".info";

    private ActionsHolder actionsHolder;
    private RequestConverter requestConverter;
    private ActionDocumentationProvider actionDocumentationProvider;

    private String applicationName;

    public void init() {
        actionDocumentationProvider = new ActionDocumentationProvider(requestConverter, ActionRouter.JSON_OM);
    }

    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        ResponseFormat responseFormat = ResponseFormat.getResponseFormat(target);
        if (responseFormat != ResponseFormat.INFO) {
            return;
        }

        baseRequest.setHandled(true);
        showInfo(target, baseRequest, request, response);
    }

    private void showInfo(String target, Request baseRequest, HttpServletRequest request,
                          HttpServletResponse response) throws IOException {
        String actionOrPrefix = target.substring(0, target.length() - ResponseFormat.INFO.getExtension().length());
        List<String> actionsList = actionsHolder.prefixes.get(actionOrPrefix);
        if (actionsList != null) {
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType("text/html");

            PrintWriter writer = response.getWriter();
            writer.print("<!DOCTYPE HTML>" +
                    "<html>" +
                    "<head><title>");
            writer.print("Methods for prefix '" + actionOrPrefix + "'");
            writer.print("</title></head>" +
                    "<body>");
            writer.print("<a href='/" + EXTENSION + "'>Go to root</a><br>");
            writer.print("Methods for prefix " + makeNavigationForPath(actionOrPrefix));
            writer.print("<ul>");
            for (String action : actionsList) {
                writer.print("<li>" + makeNavigationForPath(action) + "</li>");
            }
            writer.print("</ul>" +
                    "</body>" +
                    "</html>");
            writer.flush();
            return;
        }

        Action<?, ?> action = actionsHolder.actions.get(actionOrPrefix);
        if (action != null) {
            ActionParameters actionParameters = ActionReflectionUtils.getActionParameters(action.getClass());
            response.setStatus(HttpServletResponse.SC_OK);
            response.setContentType(HTML_MARKUP_RENDERER.contentType());
            response.setCharacterEncoding("utf8");

            PrintWriter writer = response.getWriter();
            Markup markup = actionDocumentationProvider.makeDocumentationMarkup(actionOrPrefix,
                    action,
                    actionsHolder.reqClass2reqFilters.get(actionParameters.requestClass),
                    actionsHolder.registeredResponseFilters);
            String title = actionOrPrefix + " - " + applicationName + " documentation";
            String html = HTML_MARKUP_RENDERER.render(markup, title, 2, 2);
            writer.print(html);

            writer.flush();
            return;
        }

        //error 404
        response.setStatus(HttpServletResponse.SC_NOT_FOUND);
        response.setContentType("text/html");

        PrintWriter writer = response.getWriter();
        writer.print("<!DOCTYPE HTML>" +
                "<html>" +
                "<head><title>");
        writer.print("Error 404: Methods for prefix '" + actionOrPrefix + "' not found");
        writer.print("</title></head>" +
                "<body>");
        writer.print("Error 404: Methods for prefix '" + actionOrPrefix + "' not found");
        writer.print("<a href='/" + EXTENSION + "'>Go to root</a><br>");
        writer.print("</body>" +
                "</html>");
        writer.flush();
    }

    private String makeNavigationForPath(String path) {
        StringBuilder result = new StringBuilder();
        StringBuilder fullPath = new StringBuilder();
        StringBuilder lastPart = new StringBuilder();
        for (int i = 0; i < path.length(); i++) {
            char c = path.charAt(i);
            fullPath.append(c);
            if (c != '/') {
                lastPart.append(c);
            }
            if (i == path.length() - 1 || c == '/') {
                if (lastPart.length() > 0) {
                    result.append("<a href='").append(fullPath).append(EXTENSION).append("'>").append(lastPart).append("</a>");
                }
                if (c == '/') {
                    result.append("/");
                }
                lastPart.setLength(0);
            }
        }
        return result.toString();
    }

    @Required
    public void setActionsHolder(ActionsHolder actionsHolder) {
        this.actionsHolder = actionsHolder;
    }

    @Required
    public void setRequestConverter(RequestConverter requestConverter) {
        this.requestConverter = requestConverter;
    }

    @Required
    public void setApplicationName(String applicationName) {
        this.applicationName = applicationName;
    }
}
