package ru.yandex.webmaster3.core.http;

import java.io.IOException;
import java.io.OutputStream;
import java.io.PrintWriter;
import java.io.StringWriter;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;

import javax.servlet.MultipartConfigElement;
import javax.servlet.ServletException;
import javax.servlet.ServletOutputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.datastax.driver.core.utils.UUIDs;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.ser.FilterProvider;
import com.fasterxml.jackson.databind.ser.impl.SimpleBeanPropertyFilter;
import com.fasterxml.jackson.databind.ser.impl.SimpleFilterProvider;
import com.fasterxml.jackson.dataformat.xml.XmlMapper;
import com.fasterxml.jackson.datatype.jdk8.Jdk8Module;
import com.fasterxml.jackson.datatype.joda.JodaModule;
import org.apache.http.HttpStatus;
import org.eclipse.jetty.server.Request;
import org.eclipse.jetty.server.handler.AbstractHandler;
import org.jetbrains.annotations.NotNull;
import org.joda.time.DateTime;
import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.slf4j.MDC;
import org.springframework.beans.factory.annotation.Required;

import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.http.internal.ActionParameters;
import ru.yandex.webmaster3.core.http.internal.ActionReflectionUtils;
import ru.yandex.webmaster3.core.http.request.RequestContext;
import ru.yandex.webmaster3.core.http.request.RequestId;
import ru.yandex.webmaster3.core.http.request.RequestIdAware;
import ru.yandex.webmaster3.core.metrics.MonitoringCategoryUtil;
import ru.yandex.webmaster3.core.solomon.metric.SolomonKey;
import ru.yandex.webmaster3.core.solomon.metric.SolomonMetricRegistry;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimer;
import ru.yandex.webmaster3.core.solomon.metric.SolomonTimerConfiguration;
import ru.yandex.webmaster3.core.tracer.BeautyYdbTrace;
import ru.yandex.webmaster3.core.tracer.YdbMetricsService;
import ru.yandex.webmaster3.core.tracer.YdbTracer;
import ru.yandex.webmaster3.core.util.joda.jackson.WebmasterDurationModule;
import ru.yandex.webmaster3.core.util.json.FrontendAnnotationIntrospector;
import ru.yandex.webmaster3.core.util.json.JsonMapping;

/**
 * @author aherman
 */
public class ActionRouter extends AbstractHandler {
    private static final Logger log = LoggerFactory.getLogger(ActionRouter.class);
    private static final Logger actionLog = LoggerFactory.getLogger("ActionResult");
    private static final String REQUEST_ID_KEY = "actionRouter_requestId";
    protected static final String SOLOMON_LABEL_ACTION = "action";
    protected static final String SOLOMON_LABEL_CATEGORY = "category";
    protected static final String SOLOMON_LABEL_RESULT = "result";

    private ActionsHolder actionsHolder;

    private Map<String, ActionMetrics> metricsMap;

    private String applicationName;
    private String applicationVersion;
    private String applicationTmpFolder;
    private Long requestTimeout;
    private boolean enableRequestTrace = true;

    private RequestConverter requestConverter;

    private int maxRequestFileSize = 10 * 1024 * 1024; // 10 Mb

    protected SolomonTimerConfiguration solomonTimerConfiguration;
    protected SolomonMetricRegistry solomonMetricRegistry;

    private static final String ENABLE_PRETTY = "_pretty";
    private static final String ENABLE_QUERY_TRACE = "_qtrace";

    static final ObjectMapper JSON_OM = new ObjectMapper()
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
            .enable(SerializationFeature.WRITE_EMPTY_JSON_ARRAYS)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .registerModule(new JodaModule())
            .registerModule(new WebmasterJsonModule(true))
            .registerModule(new Jdk8Module())
            .registerModule(new WebmasterDurationModule(false))
            .setAnnotationIntrospector(new FrontendAnnotationIntrospector());

    private static final ObjectMapper JSON_PRETTY_OM = new ObjectMapper()
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
            .enable(SerializationFeature.INDENT_OUTPUT)
            .registerModule(new JodaModule())
            .registerModule(new WebmasterJsonModule(true))
            .registerModule(new Jdk8Module())
            .registerModule(new WebmasterDurationModule(false))
            .setAnnotationIntrospector(new FrontendAnnotationIntrospector());

    private static final ObjectMapper XML_OM = new XmlMapper()
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
            .disable(SerializationFeature.WRITE_EMPTY_JSON_ARRAYS)
            .setSerializationInclusion(JsonInclude.Include.NON_NULL)
            .registerModule(new JodaModule())
            .registerModule(new WebmasterJsonModule(true))
            .registerModule(new Jdk8Module())
            .registerModule(new WebmasterDurationModule(false));

    private static final ObjectMapper XML_PRETTY_OM = new XmlMapper()
            .disable(SerializationFeature.FAIL_ON_EMPTY_BEANS)
            .disable(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS)
//            .enable(SerializationFeature.INDENT_OUTPUT)
            .registerModule(new JodaModule())
            .registerModule(new WebmasterJsonModule(true))
            .registerModule(new Jdk8Module())
            .registerModule(new WebmasterDurationModule(false));

    // GDPR
    private static final SimpleBeanPropertyFilter USER_ID_FILTER = SimpleBeanPropertyFilter.serializeAllExcept("userId");
    private static final FilterProvider OM_FILTERS = new SimpleFilterProvider().addFilter("USER_ID_FILTER", USER_ID_FILTER);

    private static final Timer ACTION_TIMER = new Timer("ActionRouterTimer", true);

    private static final String ACTION_RESULT_LOGGING = "Action Result: status={} action={} timeMs={}";

    public void init() {
        metricsMap = new HashMap<>(actionsHolder.actions.size());
        for (Map.Entry<String, Action> actionEntry : actionsHolder.actions.entrySet()) {
            String category = MonitoringCategoryUtil.getCategory(actionEntry.getValue().getClass()).orElse(null);
            ActionMetrics metrics = new ActionMetrics(actionEntry.getKey(), category);
            metricsMap.put(actionEntry.getKey(), metrics);
        }
    }

    @Override
    public void handle(String target, Request baseRequest, HttpServletRequest request, HttpServletResponse response)
            throws IOException, ServletException {
        ResponseFormat responseFormat = ResponseFormat.getResponseFormat(target);
        if (responseFormat != ResponseFormat.JSON && responseFormat != ResponseFormat.XML
                && responseFormat != ResponseFormat.CSV) {
            return;
        }

        baseRequest.setHandled(true);
        if (responseFormat == ResponseFormat.JSON || responseFormat == ResponseFormat.XML) {
            response.addHeader("Access-Control-Allow-Origin", "*");
        }

        processRequest(target, baseRequest, request, response, responseFormat);
    }

    private void processRequest(String target, Request baseRequest, HttpServletRequest request,
                                HttpServletResponse response, ResponseFormat responseFormat) throws IOException {
        boolean enableQueryTrace = request.getParameter(ENABLE_QUERY_TRACE) != null;

        RequestTracer.startTrace(enableQueryTrace);
        YdbTracer.startTrace();
        ActionStatus actionStatus = ActionStatus.SUCCESS;

        if (enableRequestTrace) {
            log.info("Start request: target={}", target);
        }

        String suffix = responseFormat.getExtension();
        String actionName;
        if (target.endsWith(suffix)) {
            actionName = target.substring(0, target.length() - suffix.length());
        } else {
            actionName = target;
        }

        Action<? super ActionRequest, ? extends ActionResponse> action = actionsHolder.actions.get(actionName);
        if (action == null) {
            sendError(HttpServletResponse.SC_NOT_FOUND, target, actionName, response, responseFormat, 0L, "");
            return;
        }

        if (enableRequestTrace) {
            log.info("Request action: target={} action={} class={}", target, actionName, action.getClass().getName());
        }

        ActionParameters actionParameters = ActionReflectionUtils.getActionParameters(action.getClass());
        if (actionParameters == null) {
            sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, target, actionName, response,
                    responseFormat, 0L, "Unable to get request/response classes");
            return;
        }

        ActionRequest actualRequest = null;
        ActionResponse actualResponse = null;
        ActionResponse.ErrorResponse actionError = null;
        boolean hasException = false;

        try {
            actualRequest = actionParameters.requestClass.newInstance();
        } catch (Exception e) {
            actionStatus = ActionStatus.FAIL;
            String message = "Unable to instantiate request/response: " + e.getMessage();
            log.error(message, e);

            StringWriter sw = new StringWriter();
            e.printStackTrace(new PrintWriter(sw, true));
            actionError = new WebmasterErrorResponse.UnableToInstantateRequestResponse(this.getClass());
        }

        // GDPR
        String queryStr = request.getQueryString();
        if (queryStr != null) {
            queryStr = queryStr.replaceAll("&?userId=\\d+", "");
        }

        if (actionError == null) {
            try {
                if (enableRequestTrace) {
                    log.info("Request query: action={} query={}", actionName, queryStr);
                }
                baseRequest.setAttribute(Request.__MULTIPART_CONFIG_ELEMENT,
                        new MultipartConfigElement(applicationTmpFolder,
                                maxRequestFileSize,
                                maxRequestFileSize,
                                maxRequestFileSize)
                );
                baseRequest.extractParameters();
                actionError = requestConverter.fillRequest(actualRequest, request);
                if (actionError != null) {
                    actionStatus = actionError.getRequestStatus();
                } else {
                    if (actualRequest.isLoggable() && enableRequestTrace) {
                        String paramsStr = JSON_OM.writer(OM_FILTERS).writeValueAsString(actualRequest);
                        log.info("Request: action={} parameters={}", actionName, paramsStr);
                    } else {
                        if (enableRequestTrace) {
                            log.info("Request: action={}", actionName);
                        }
                    }
                }
            } catch (WebmasterException e) {
                log.error("Unable to read request", e);
                actionStatus = e.getError().getRequestStatus();
                actionError = e.getError();
            } catch (Exception e) {
                log.error("Unable to read request", e);
                sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, target, actionName, response, responseFormat, 0L,
                        "Unable to read request");
                return;
            }
        }

        //get hostname and ip
        if (actualRequest instanceof HostNameAwareActionRequest) {
            String ipAddress = request.getHeader("X-FORWARDED-FOR-Y");
            if (ipAddress == null) {
                ipAddress = request.getHeader("X-FORWARDED-FOR");
            }
            if (ipAddress == null) {
                ipAddress = request.getRemoteAddr();
            }

            if (ipAddress == null) {
                log.warn("Could not specify caller ip");
            } else {
                String hostName = null;
                try {
                    InetAddress inetAddress = InetAddress.getByName(ipAddress);
                    hostName = inetAddress.getHostName();
                } catch (UnknownHostException ex) {
                    log.warn("Could not specify caller hostname, ip - {}", ipAddress);
                }
                ((HostNameAwareActionRequest) actualRequest).setHostName(hostName);
            }
        }

        RequestId requestId = null;
        if (actionError == null) {
            if (actualRequest instanceof RequestIdAware) {
                RequestIdAware requestIdAware = (RequestIdAware) actualRequest;
                if (requestIdAware.getBalancerRequestId() == null) {
                    requestIdAware.setBalancerRequestId(new RequestId.FakeRequestId(UUIDs.timeBased()));
                }
                requestId = requestIdAware.getBalancerRequestId();
            }
        }
        if (requestId == null) {
            requestId = new RequestId.FakeRequestId(UUIDs.timeBased());
        }

        boolean wasInterrupted = false;
        try (MDC.MDCCloseable ignored = MDC.putCloseable(REQUEST_ID_KEY, requestId.toString())) {
            if (actionError == null) {
                try {
                    RequestContext requestContext = new RequestContext(baseRequest, action, actionName);
                    for (RequestFilter filter : actionsHolder.reqClass2reqFilters.get(actionParameters.requestClass)) {
                        actionError = filter.beforeRequest(requestContext, actualRequest);
                        wasInterrupted |= Thread.interrupted();
                        if (actionError != null) {
                            break;
                        }
                    }
                } catch (WebmasterException e) {
                    wasInterrupted |= Thread.interrupted();
                    log.warn("Got webmaster common error", e);
                    actionStatus = e.getError().getRequestStatus();
                    actionError = e.getError();
                    hasException = true;
                } catch (Exception e) {
                    wasInterrupted |= Thread.interrupted();
                    log.error("Unable to process request: name={}", actionName, e);
                    actionStatus = ActionStatus.FAIL;

                    StringWriter sw = new StringWriter();
                    e.printStackTrace(new PrintWriter(sw, true));
                    actionError = new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), e.getMessage());
                    hasException = true;
                }
            }

            if (actionError == null) {
                ActionRequestReaperTask timerTask = new ActionRequestReaperTask(Thread.currentThread());
                try {
                    if (requestTimeout != null && requestTimeout > 0) {
                        ACTION_TIMER.schedule(timerTask, requestTimeout);
                    }
                    ActionResponse result = action.process(actualRequest);
                    wasInterrupted |= Thread.interrupted();
                    if (result instanceof ActionResponse.ErrorResponse) {
                        actionError = (ActionResponse.ErrorResponse) result;
                    } else {
                        actualResponse = result;
                    }

                    List<Class<? extends ResponseFilter>> responseFilters = action.getResponseFilters();
                    for (Class<? extends ResponseFilter> filterClass : responseFilters) {
                        ResponseFilter filter = actionsHolder.registeredResponseFilters.get(filterClass);
                        if (filter != null) {
                            filter.beforeResponse(actionName, actualRequest, result);
                            wasInterrupted |= Thread.interrupted();
                        }
                    }
                } catch (WebmasterException e) {
                    wasInterrupted |= Thread.interrupted();
                    actionStatus = e.getError().getRequestStatus();
                    log.error("Got webmaster common error", e);
                    actionError = e.getError();
                } catch (Exception e) {
                    wasInterrupted |= Thread.interrupted();
                    actionStatus = ActionStatus.FAIL;
                    String message = "Unable to process request: name=" + actionName + " query=" + queryStr;
                    log.error(message, e);
                    if (timerTask.isExecuted()) {
                        actionError = new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), "Request timeout");
                    } else {
                        actionError = new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), e.getMessage());
                    }
                } finally {
                    timerTask.cancel();
                }
            }

            RequestTrace requestTrace = RequestTracer.stopTrace();
            var ydbTrace = YdbTracer.stopTrace();
            long durationMillis = requestTrace.getRequestDuration(TimeUnit.MILLISECONDS);

            if (action.getClass().isAnnotationPresent(RequestTimeout.class)) {
                RequestTimeout requestTimeout = action.getClass().getAnnotation(RequestTimeout.class);
                if (requestTimeout.value() < durationMillis) {
                    log.warn("Request timeout: name={}", actionName);
                }
            }
            updateMetrics(actualRequest, actionName, actionError, durationMillis);

            boolean isUndecoratedResponse = false;
            if (actualResponse instanceof ActionResponse.NormalResponse) {
                isUndecoratedResponse = ((ActionResponse.NormalResponse) actualResponse).isUndecoratedResponse();
            }

            Object genericActionResponse;
            if (isUndecoratedResponse) {
                genericActionResponse = actualResponse;
            } else {
                genericActionResponse = new GenericActionResponse(
                        applicationName,
                        applicationVersion,
                        DateTime.now(),
                        actionName,
                        actionStatus,
                        durationMillis,
                        actualRequest,
                        actualResponse,
                        actionError,
                        requestTrace
                );
            }
            if (ydbTrace != null) {
                YdbMetricsService.saveSolomon(ydbTrace);

                if (enableRequestTrace) {
                    log.info("Request YDB trace pretty: {} {}", target, new BeautyYdbTrace(ydbTrace));
                    log.info("Request YDB trace json: {} {}", target, JsonMapping.writeValueAsString(ydbTrace));
                }
            }

            if (enableRequestTrace) {
                log.info("Request trace: {} {} {} cr {} crt {} cw {} cwt {} cb {} cbt {} chr {} chrt {} chw {} chwt {} yr {} yrt {} yw {} ywt {}",
                        target,
                        requestTrace.getRequestDuration(TimeUnit.MILLISECONDS),
                        requestTrace.getAllQueriesDuration(TimeUnit.MILLISECONDS),
                        requestTrace.getCassandraReadCount(), requestTrace.getTotalCassandraReadTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getCassandraWriteCount(), requestTrace.getTotalCassandraWriteTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getCassandraBatchCount(), requestTrace.getTotalCassandraBatchTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getClickhouseReadCount(), requestTrace.getTotalClickhouseReadTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getClickhouseWriteCount(), requestTrace.getTotalClickhouseWriteTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getYdbReadCount(), requestTrace.getTotalYdbReadTimeNano(TimeUnit.MILLISECONDS),
                        requestTrace.getYdbWriteCount(), requestTrace.getTotalYdbWriteTimeNano(TimeUnit.MILLISECONDS)

                );
                log.info("Finish request: status={} target={} action={} httpCode={} timeMs={}", actionStatus, target, actionName,
                        response.getStatus(), durationMillis);
            }

            final String actionResult;
            if (actionError == null) {
                actionResult = "SUCCESS";
            } else {
                actionResult = actionError.getSubsystem() == WebmasterCommonErrorSystem.INTERNAL ? "FAIL" : "SUCCESS";
            }

            if (enableRequestTrace) {
                actionLog.info(ACTION_RESULT_LOGGING, actionResult, actionName, durationMillis);
            }

            boolean prettyPrint = request.getParameter(ENABLE_PRETTY) != null;

            response.setCharacterEncoding("UTF-8");
            response.setContentType(responseFormat.getMimeType());
            if (actionError != null && responseFormat == ResponseFormat.CSV) {
                int status = HttpStatus.SC_NOT_FOUND;
                response.setStatus(status);
                try (PrintWriter pw = response.getWriter()) {
                    pw.write("Not found");
                }

                if (enableRequestTrace) {
                    log.info("Response sent: status={} target={} action={}", status, target, actionName);
                }
            } else if (actualResponse instanceof BinaryActionResponse) {
                int status = ((BinaryActionResponse) actualResponse).getStatus();
                response.setStatus(status);
                try (OutputStream os = response.getOutputStream()) {
                    os.write(((BinaryActionResponse) actualResponse).getResponseData());
                }

                if (enableRequestTrace) {
                    log.info("Response sent: status={} target={} action={}", status, target, actionName);
                }
            } else {
                response.setStatus(HttpServletResponse.SC_OK);

                ObjectMapper om = JSON_OM;
                if (responseFormat == ResponseFormat.JSON) {
                    if (prettyPrint || actionError != null) {
                        om = JSON_PRETTY_OM;
                    } else {
                        om = JSON_OM;
                    }

                } else if (responseFormat == ResponseFormat.XML) {
                    if (prettyPrint || actionError != null) {
                        om = XML_PRETTY_OM;
                    } else {
                        om = XML_OM;
                    }
                }

                ServletOutputStream os = null;
                try {
                    os = response.getOutputStream();
                    om.writeValue(os, genericActionResponse);
                } finally {
                    if (os != null) {
                        os.flush();
                    }
                }

                if (enableRequestTrace) {
                    log.info("Response sent: target={} action={}", target, actionName);
                }
            }
        } finally {
            if (wasInterrupted) {
                // Прикольная магия - логирование при включенном interrupt-флаге - просто не работает.
                // Если где-то в action'е случится interrupt - то сюда мы придем с каким-нибудь exception'ом (не InterruptedException)
                // и включеным interrupt-флагом - что означает, что все посмертное логирование остановленного action'а
                // будет просто продолбано. Поэтому по телу метода сбрасываем флаг после каждого exception'а,
                // и восстанавливаем в конце
                log.error("This thread was interrupted");
                Thread.currentThread().interrupt();
            }
        }
    }

    protected void updateMetrics(ActionRequest request, String actionName, ActionResponse.ErrorResponse actionError,
                                 long durationMillis) {
        ActionMetrics metrics = metricsMap.get(actionName);
        Duration duration = Duration.millis(durationMillis);
        if (actionError == null) {
            metrics.success.update(duration);
        } else if (actionError.getSubsystem() == WebmasterCommonErrorSystem.INTERNAL) {
            metrics.internalError.update(duration);
        } else {
            metrics.userError.update(duration);
        }
    }

    private void sendError(int status, String requestTarget, String actionName, HttpServletResponse response,
                           ResponseFormat responseFormat, long durationMs, String message) throws IOException {
        log.error("Finish request: status={} target={} action={} httpCode={}, message={}", ActionStatus.FAIL, requestTarget,
                actionName, status, message);

        actionLog.info(ACTION_RESULT_LOGGING, "FAIL", actionName, durationMs);

        response.setStatus(status);
        response.setContentType("text/plain");
        if (ResponseFormat.CSV != responseFormat) {
            try (PrintWriter writer = response.getWriter()) {
                writer.println(message);
            }
        } else {
            try (PrintWriter writer = response.getWriter()) {
                writer.write("Error");
            }
        }
    }

    protected class ActionMetrics {
        public final SolomonTimer success;
        public final SolomonTimer userError;
        public final SolomonTimer internalError;

        protected ActionMetrics(String action, String category) {
            SolomonKey baseKey = createBaseKey(action, category);

            this.success = solomonMetricRegistry.createTimer(
                    solomonTimerConfiguration,
                    baseKey.withLabel(SOLOMON_LABEL_RESULT, "success")
            );
            this.userError = solomonMetricRegistry.createTimer(
                    solomonTimerConfiguration,
                    baseKey.withLabel(SOLOMON_LABEL_RESULT, "user_error")
            );
            this.internalError = solomonMetricRegistry.createTimer(
                    solomonTimerConfiguration,
                    baseKey.withLabel(SOLOMON_LABEL_RESULT, "internal_error")
            );

        }

        @NotNull
        protected SolomonKey createBaseKey(String action, String category) {
            return SolomonKey.create(SOLOMON_LABEL_ACTION, action)
                    .withLabel(SOLOMON_LABEL_CATEGORY, category == null ? "<unknown>" : category);
        }


    }

    @Required
    public void setActionsHolder(ActionsHolder actionsHolder) {
        this.actionsHolder = actionsHolder;
    }

    @Required
    public void setApplicationTmpFolder(String applicationTmpFolder) {
        this.applicationTmpFolder = applicationTmpFolder;
    }

    public void setRequestTimeout(Long requestTimeout) {
        this.requestTimeout = requestTimeout;
    }

    public void setMaxRequestFileSize(int maxRequestFileSize) {
        this.maxRequestFileSize = maxRequestFileSize;
    }

    @Required
    public void setApplicationName(String applicationName) {
        this.applicationName = applicationName;
    }

    @Required
    public void setApplicationVersion(String applicationVersion) {
        this.applicationVersion = applicationVersion;
    }

    @Required
    public void setRequestConverter(RequestConverter requestConverter) {
        this.requestConverter = requestConverter;
    }

    @Required
    public void setSolomonTimerConfiguration(SolomonTimerConfiguration solomonTimerConfiguration) {
        this.solomonTimerConfiguration = solomonTimerConfiguration;
    }

    @Required
    public void setSolomonMetricRegistry(SolomonMetricRegistry solomonMetricRegistry) {
        this.solomonMetricRegistry = solomonMetricRegistry;
    }

    public void setEnableRequestTrace(boolean enableRequestTrace) {
        this.enableRequestTrace = enableRequestTrace;
    }

    public static class ActionRequestReaperTask extends TimerTask {

        private final Thread actionThread;
        private volatile boolean executed = false;

        public ActionRequestReaperTask(Thread actionThread) {
            this.actionThread = actionThread;
        }

        @Override
        public void run() {
            if (!executed) {
                log.error("Interrupting thread " + actionThread.getName());
                executed = true;
                actionThread.interrupt();
            }
        }

        public boolean isExecuted() {
            return executed;
        }
    }
}
