package ru.yandex.webmaster3.core.http;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.autodoc.common.doc.DocUtils;
import ru.yandex.autodoc.common.doc.types.EnumType;
import ru.yandex.autodoc.common.doc.types.ValueType;
import ru.yandex.webmaster3.core.util.enums.EnumResolver;
import ru.yandex.webmaster3.core.util.ReflectionUtils;

import java.lang.reflect.Field;
import java.lang.reflect.Type;
import java.util.ArrayList;
import java.util.List;

/**
 * @author aherman
 */
public class EnumParameterConverter implements ParameterConverter {
    private static final Logger log = LoggerFactory.getLogger(EnumParameterConverter.class);

    @Override
    public Object convert(String value, Type toType) {
        Class<?> enumClazz = isEnum(toType);
        if (enumClazz != null) {
            return getEnum(value, enumClazz);
        }
        return null;
    }

    @Override
    public ValueType describeType(Type type) {
        Class<? extends Enum<?>> enumClass = isEnum(type);
        if (enumClass != null) {
            List<EnumType.Entry> entries = new ArrayList<>();
            for (Enum<?> value : enumClass.getEnumConstants()) {
                String name = value.name();
                String desc = null;
                try {
                    desc = DocUtils.getDescriptionForAnnotatedElement(enumClass.getField(value.name()));
                } catch (NoSuchFieldException e) {
                    log.error("Failed to get description for enum entry " + name + " of class " + enumClass);
                }
                entries.add(new EnumType.Entry(name, desc));
            }
            return new EnumType(enumClass.getSimpleName(), entries);
        }
        return null;
    }

    private static Object getEnum(String valueStr, Type type) {
        Class<?> desiredClass = ReflectionUtils.getClassFromType(type);
        try {
            Field r = desiredClass.getField("R");
            if (EnumResolver.class.isAssignableFrom(r.getType())) {
                EnumResolver<?> enumResolver = (EnumResolver) r.get(null);
                Object result = enumResolver.valueOfOrNull(valueStr);
                if (result == null) {
                    result = enumResolver.getUnknownValue();
                }
                if (result == null) {
                    throw new IllegalArgumentException("Not found enum value \"" + valueStr + "\"");
                }
                return result;
            }
        } catch (NoSuchFieldException | IllegalAccessException e) {
            // Resolver is not found. Try to find exact enum value.
        }

        Enum[] enumConstants = (Enum[]) desiredClass.getEnumConstants();
        for (Enum enumConstant : enumConstants) {
            if (enumConstant.name().equalsIgnoreCase(valueStr)) {
                return enumConstant;
            }
        }
        throw new IllegalArgumentException("Not found enum value " + valueStr);
    }

    private static Class<? extends Enum<?>> isEnum(Type type) {
        Class<?> clazz = ReflectionUtils.getClassFromType(type);
        if (clazz.isEnum()) {
            return (Class<? extends Enum<?>>) clazz;
        } else {
            return null;
        }
    }
}
