package ru.yandex.webmaster3.core.http;

import java.io.BufferedInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.zip.GZIPInputStream;

import javax.servlet.http.Part;

import com.fasterxml.jackson.annotation.JsonIgnore;
import org.apache.commons.io.IOUtils;
import ru.yandex.webmaster3.core.WebmasterException;

/**
 * @author aherman
 */
public class FileParameter {
    private static final String MIME_APPLICATION_X_GZIP = "application/x-gzip";

    private final Part part;

    public FileParameter(Part part) {
        this.part = part;
    }

    @JsonIgnore
    public InputStream getInputStream() throws IOException {
        InputStream inputStream = part.getInputStream();
        if (MIME_APPLICATION_X_GZIP.equals(part.getContentType())) {
            inputStream = new GZIPInputStream(inputStream);
        }
        return inputStream;
    }

    @JsonIgnore
    public byte[] getRawData(int maxLength) {
        ByteArrayOutputStream baos = new ByteArrayOutputStream(24 * 1024);
        try {
            BufferedInputStream bufferedInputStream = new BufferedInputStream(part.getInputStream(), 8);
            IOUtils.copyLarge(bufferedInputStream, baos, 0, maxLength);
        } catch (IOException e) {
            throw new WebmasterException("Failed to read data from file",
                    new WebmasterErrorResponse.InternalUnknownErrorResponse(this.getClass(), ""), e);
        }
        return baos.toByteArray();
    }

    public String getContentType() {
        return part.getContentType();
    }

    public String getName() {
        return part.getName();
    }

    public String getFileName() {
        String fileName = "";
        String header = part.getHeader("Content-Disposition");
        if (header != null) {
            String[] values = header.split(";");
            for (String value : values) {
                if (value.trim().startsWith("filename=")) {
                    String temp = value.trim().substring("fileName=".length());
                    fileName = temp.trim().replace("\"", "");
                    break;
                }
            }
        }
        return fileName;
    }

    public long getSize() {
        return part.getSize();
    }
}
