package ru.yandex.webmaster3.core.http.autodoc;

import ru.yandex.webmaster3.core.util.ReflectionUtils;

import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.lang.reflect.WildcardType;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;

/**
 * @author avhaliullin
 */
public class FullTypeInfo {
    private final Class<?> clazz;
    private final Type originalType;
    private Map<TypeVariable, FullTypeInfo> generics;
    private List<FullTypeInfo> upperBounds;
    private final List<FullTypeInfo> lowerBounds;

    public FullTypeInfo(Class<?> clazz, Type originalType, Map<TypeVariable, FullTypeInfo> generics) {
        this.clazz = clazz;
        this.originalType = originalType;
        this.generics = generics;
        this.upperBounds = Collections.singletonList(this);
        this.lowerBounds = Collections.singletonList(this);
    }

    public FullTypeInfo(Type originalType, List<FullTypeInfo> upperBounds, List<FullTypeInfo> lowerBounds) {
        this.clazz = null;
        this.originalType = originalType;
        this.generics = Collections.emptyMap();
        this.upperBounds = upperBounds;
        this.lowerBounds = lowerBounds;
    }

    public Map<TypeVariable, FullTypeInfo> getGenerics() {
        return generics;
    }

    public List<FullTypeInfo> getGenericsList() {
        if (!isConcrete()) {
            return Collections.emptyList();
        }
        List<FullTypeInfo> result = new ArrayList<>();
        for (TypeVariable tv : clazz.getTypeParameters()) {
            result.add(generics.get(tv));
        }
        return result;
    }

    public Class<?> getClazz() {
        if (!isConcrete()) {
            throw new RuntimeException("This type is not concrete: " + originalType.getTypeName());
        }
        return clazz;
    }

    public Type getOriginalType() {
        return originalType;
    }

    public FullTypeInfo memberFullType(Type type) {
        if (type instanceof Class) {
            return createSimple((Class) type);
        } else if (type instanceof ParameterizedType) {
            ParameterizedType parameterizedType = (ParameterizedType) type;
            Class<?> memberClass = (Class<?>) parameterizedType.getRawType();
            Map<TypeVariable, FullTypeInfo> variables = new LinkedHashMap<>();
            for (int i = 0; i < memberClass.getTypeParameters().length; i++) {
                TypeVariable tv = memberClass.getTypeParameters()[i];
                Type tvValue = parameterizedType.getActualTypeArguments()[i];
                variables.put(tv, memberFullType(tvValue));
            }
            return new FullTypeInfo(memberClass, type, variables);
        } else if (type instanceof TypeVariable) {
            FullTypeInfo result = generics.get(type);
            if (result == null) {
                TypeVariable tv = (TypeVariable) type;
                return new FullTypeInfo(tv, resolveBounds(tv.getBounds()), Collections.emptyList());
            }
            return result;
        } else if (type instanceof WildcardType) {
            WildcardType wt = (WildcardType) type;
            return new FullTypeInfo(wt, resolveBounds(wt.getUpperBounds()), resolveBounds(wt.getLowerBounds()));
        } else {
            throw new RuntimeException("Cannot extract full type info for type " + type + " as member of " + this + ": unknown type kind");
        }
    }

    private List<FullTypeInfo> resolveBounds(Type[] bounds) {
        List<FullTypeInfo> result = new ArrayList<>();
        for (Type bound : bounds) {
            result.add(memberFullType(bound));
        }
        return result;
    }

    public FullTypeInfo ancestorFullType(Class<?> ancestorClass) {
        if (!isConcrete()) {
            throw new RuntimeException("Cannot get ancestor full type of non-concrete type " + originalType.getTypeName());
        }
        if (!ancestorClass.isAssignableFrom(clazz)) {
            throw new RuntimeException("Cannot extract full type info for type " + ancestorClass + ": not an ancestor of " + this);
        }

        if (ancestorClass == clazz) {
            return this;
        }

        if (ancestorClass.getTypeParameters().length == 0) {
            return createSimple(ancestorClass);
        }

        Type parent = clazz.getGenericSuperclass();
        if (parent == null || !ancestorClass.isAssignableFrom(ReflectionUtils.getClassFromType(parent))) {
            for (Type p : clazz.getGenericInterfaces()) {
                if (ancestorClass.isAssignableFrom(ReflectionUtils.getClassFromType(p))) {
                    parent = p;
                    break;
                }
            }
        }

        FullTypeInfo directAncestorFullType = memberFullType(parent);
        if (directAncestorFullType.clazz == ancestorClass) {
            return directAncestorFullType;
        } else {
            return directAncestorFullType.ancestorFullType(ancestorClass);
        }
    }

    public FullTypeInfo thisOrUpper() {
        if (isConcrete()) {
            return this;
        }
        if (!upperBounds.isEmpty()) {
            return upperBounds.get(0);
        }
        throw new RuntimeException("Type " + this + " neither concrete or have upper bounds");
    }

    public FullTypeInfo thisOrAnyBound() {
        if (isConcrete()) {
            return this;
        }
        if (!lowerBounds.isEmpty()) {
            return lowerBounds.get(0);
        }
        if (!upperBounds.isEmpty()) {
            return upperBounds.get(0);
        }
        throw new RuntimeException("Type " + this + " neither concrete or have upper/lower bounds");
    }

    public FullTypeInfo thisOrLower() {
        if (isConcrete()) {
            return this;
        }
        if (!lowerBounds.isEmpty()) {
            return lowerBounds.get(0);
        }
        throw new RuntimeException("Type " + this + " neither concrete or have lower bounds");
    }

    public List<FullTypeInfo> getUpperBounds() {
        return upperBounds;
    }

    public List<FullTypeInfo> getLowerBounds() {
        return lowerBounds;
    }

    public boolean isConcrete() {
        return clazz != null;
    }

    public boolean isAssignableFrom(Class that) {
        if (isConcrete()) {
            return clazz.isAssignableFrom(that);
        }
        for (FullTypeInfo upper : upperBounds) {
            if (upper.isConcrete() && !upper.getClazz().isAssignableFrom(that)) {
                return false;
            }
        }
        return true;
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        FullTypeInfo that = (FullTypeInfo) o;

        if (!isConcrete()) {
            return originalType.equals(that.originalType) &&
                    this.upperBounds.equals(that.upperBounds) &&
                    this.lowerBounds.equals(that.lowerBounds);
        }
        if (!clazz.equals(that.clazz)) {
            return false;
        }
        return generics.equals(that.generics);

    }

    @Override
    public int hashCode() {
        if (isConcrete()) {
            int result = clazz.hashCode();
            result = 31 * result + generics.hashCode();
            return result;
        } else {
            int result = originalType.hashCode();
            result += upperBounds.size();
            for (FullTypeInfo lowerBound : lowerBounds) {
                result += lowerBound.hashCode();
            }
            return result;
        }
    }

    @Override
    public String toString() {
        StringBuilder sb = new StringBuilder(clazz.getCanonicalName());
        if (isConcrete()) {
            if (!generics.isEmpty()) {
                sb.append("<");
                boolean first = true;
                for (FullTypeInfo x : generics.values()) {
                    if (!first) {
                        sb.append(", ");
                    }
                    first = false;
                    sb.append(x.toString());
                }
                sb.append(">");
            }
        } else {
            if (originalType != null && originalType instanceof TypeVariable<?>) {
                sb.append(((TypeVariable) originalType).getName());
            } else {
                sb.append("?");
            }
            String delim = " extends ";
            for (FullTypeInfo upper : upperBounds) {
                sb.append(delim);
                delim = " & ";
                sb.append(upper);
            }
            delim = " super ";
            for (FullTypeInfo lower : lowerBounds) {
                sb.append(delim);
                delim = " & ";
                sb.append(lower);
            }
        }
        return sb.toString();
    }

    public static FullTypeInfo createFromAny(Type type) {
        Map<Type, FullTypeInfo> typesCache = new HashMap<>();
        return createFromAny(type, typesCache);
    }

    public static FullTypeInfo createFromAny(Type type, Map<Type, FullTypeInfo> typesCache) {
        if (type instanceof ParameterizedType) {
            return createParameterized((ParameterizedType) type, typesCache);
        } else if (type instanceof Class<?>) {
            return createSimple((Class) type);
        } else if (type instanceof TypeVariable) {
            return createVariable((TypeVariable) type, typesCache);
        } else if (type instanceof WildcardType) {
            return createWildcard((WildcardType) type, typesCache);
        } else {
            throw new RuntimeException("Cannot extract full type info for type " + type + ": unknown type kind");
        }
    }

    public static FullTypeInfo createParameterized(ParameterizedType type, Map<Type, FullTypeInfo> typesCache) {
        Class<?> memberClass = (Class<?>) type.getRawType();
        Map<TypeVariable, FullTypeInfo> variables = new LinkedHashMap<>();
        FullTypeInfo result = new FullTypeInfo(memberClass, type, variables);
        typesCache.put(type, result);
        for (int i = 0; i < memberClass.getTypeParameters().length; i++) {
            TypeVariable tv = memberClass.getTypeParameters()[i];
            Type tvValue = type.getActualTypeArguments()[i];
            variables.put(tv, createFromAny(tvValue, typesCache));
        }
        result.generics = variables;
        return result;
    }

    private static FullTypeInfo createVariable(TypeVariable typeVariable, Map<Type, FullTypeInfo> typesCache) {
        FullTypeInfo result = new FullTypeInfo(typeVariable, Collections.emptyList(), Collections.emptyList());
        typesCache.put(typeVariable, result);
        result.upperBounds = prepareBoundsArray(typeVariable.getBounds(), typesCache);
        return result;
    }

    public static FullTypeInfo createWildcard(WildcardType wildcardType, Map<Type, FullTypeInfo> typesCache) {
        return new FullTypeInfo(
                wildcardType,
                prepareBoundsArray(wildcardType.getUpperBounds(), typesCache),
                prepareBoundsArray(wildcardType.getLowerBounds(), typesCache)
        );
    }

    public static FullTypeInfo createSimple(Class<?> clazz) {
        Map<TypeVariable, FullTypeInfo> variables = new HashMap<>();
        Map<Type, FullTypeInfo> typesCache = new HashMap<>(variables);
        for (int i = 0; i < clazz.getTypeParameters().length; i++) {
            TypeVariable tv = clazz.getTypeParameters()[i];
            variables.put(tv, createVariable(tv, typesCache));
        }
        return new FullTypeInfo(clazz, clazz, variables);
    }

    private static List<FullTypeInfo> prepareBoundsArray(Type[] bounds) {
        Map<Type, FullTypeInfo> cache = new HashMap<>();
        return prepareBoundsArray(bounds, cache);
    }

    private static List<FullTypeInfo> prepareBoundsArray(Type[] bounds, Map<Type, FullTypeInfo> typesCache) {
        List<FullTypeInfo> result = new ArrayList<>();
        for (Type bound : bounds) {
            FullTypeInfo boundType = typesCache.get(bound);
            if (boundType == null) {
                boundType = createFromAny(bound, typesCache);
                typesCache.put(bound, boundType);
            }
            result.add(boundType);
        }
        return result;
    }
}
