package ru.yandex.webmaster3.core.http.internal;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.function.Function;

import ru.yandex.webmaster3.core.http.Action;
import ru.yandex.webmaster3.core.http.ActionEffect;
import ru.yandex.webmaster3.core.http.ActionRequest;
import ru.yandex.webmaster3.core.http.ActionResponse;
import ru.yandex.webmaster3.core.http.ReadAction;
import ru.yandex.webmaster3.core.http.RequestFileProperty;
import ru.yandex.webmaster3.core.http.RequestFilter;
import ru.yandex.webmaster3.core.http.RequestPostProperty;
import ru.yandex.webmaster3.core.http.RequestQueryProperty;
import ru.yandex.webmaster3.core.http.WriteAction;
import ru.yandex.webmaster3.core.http.autodoc.FullTypeInfo;
import ru.yandex.webmaster3.core.util.ReflectionUtils;

/**
 * @author avhaliullin
 */
public class ActionReflectionUtils {
    public static ActionEffect getActionEffect(Class<? extends Action> actionClass) {
        boolean annotatedRead = actionClass.isAnnotationPresent(ReadAction.class);
        boolean annotatedWrite = actionClass.isAnnotationPresent(WriteAction.class);
        if (annotatedRead && annotatedWrite) {
            throw new RuntimeException("Action annotated with both found: " + actionClass.getName());
        }
        if (annotatedRead) {
            return ActionEffect.READ;
        } else if (annotatedWrite) {
            return ActionEffect.WRITE;
        } else {
            return null;
        }
    }

    public static FullTypeInfo getRequestFilterRequestType(RequestFilter requestFilter) {
        return FullTypeInfo.createSimple(requestFilter.getClass())
                .ancestorFullType(RequestFilter.class)
                .getGenericsList()
                .get(0);
    }

    public static FullTypeInfo getRequestFilterResponseType(RequestFilter requestFilter) {
        return FullTypeInfo.createSimple(requestFilter.getClass())
                .ancestorFullType(RequestFilter.class)
                .getGenericsList()
                .get(1);
    }

    public static ActionParameters getActionParameters(Class<? extends Action> actionClass) {
        FullTypeInfo actionTypeInfo = FullTypeInfo.createSimple(actionClass).ancestorFullType(Action.class);
        return new ActionParameters(
                (Class<? extends ActionRequest>) actionTypeInfo.getGenericsList().get(0).getClazz(),
                (Class<? extends ActionResponse>) actionTypeInfo.getGenericsList().get(1).getClazz()
        );
    }

    public static Collection<ParameterInfo> getAllRequestParameters(Class<?> clazz) {
        return getAllRequestParameters(clazz, Function.identity());
    }

    public static Collection<ParameterInfo> getAllRequestParameters(Class<?> clazz, Function<String, String> propertyNamingStrategy) {
        FullTypeInfo requestType = FullTypeInfo.createSimple(clazz);

        Object instance = null;
        try {
            instance = ReflectionUtils.instantiateWithDefaults(clazz);
        } catch (Exception e) {
            //ignored
        }

        Map<String, List<Method>> methodsMap = new HashMap<>();
        Map<String, Field> fieldMap = new HashMap<>();
        fillSetterMethods(clazz, methodsMap, fieldMap);

        List<ParameterInfo> result = new ArrayList<>();

        for (Map.Entry<String, List<Method>> entry : methodsMap.entrySet()) {
            String internalName = entry.getKey();
            String parameterName = propertyNamingStrategy.apply(internalName);

            Annotation propTypeAnnotation = null;
            FullTypeInfo parameterType = null;
            Map<Class<? extends Annotation>, List<Annotation>> annotations = new HashMap<>();
            Method mostSpecificMethod = null;

            List<Method> methods = entry.getValue();
            final Field field = fieldMap.get(internalName);
            if (field != null) {
                for (Annotation annotation : field.getAnnotations()) {
                    annotations
                            .computeIfAbsent(annotation.annotationType(), ign -> new ArrayList<>())
                            .add(annotation);
                }
            }
            // теперь методы упорядочены сверху вниз по иерархии
            Collections.reverse(methods);
            for (Method method : methods) {
                FullTypeInfo declaringType = requestType.ancestorFullType(method.getDeclaringClass());

                RequestFileProperty fileAnnotation = method.getAnnotation(RequestFileProperty.class);
                RequestQueryProperty getAnnotation = method.getAnnotation(RequestQueryProperty.class);
                RequestPostProperty postAnnotation = method.getAnnotation(RequestPostProperty.class);
                if (fileAnnotation != null) {
                    propTypeAnnotation = fileAnnotation;
                } else if (getAnnotation != null) {
                    propTypeAnnotation = getAnnotation;
                } else if (postAnnotation != null) {
                    propTypeAnnotation = postAnnotation;
                }

                parameterType = declaringType.memberFullType(method.getGenericParameterTypes()[0]);

                for (Annotation annotation : method.getAnnotations()) {
                    annotations
                            .computeIfAbsent(annotation.annotationType(), ign -> new ArrayList<>())
                            .add(annotation);
                }
                mostSpecificMethod = method;
            }

            if (propTypeAnnotation == null) {
                continue;
            }

            boolean parameterIsRequired = false;
            ParameterInfo.PropertyType type;
            long maxSize = 0;
            if (propTypeAnnotation instanceof RequestFileProperty) {
                RequestFileProperty fileAnnotation = (RequestFileProperty) propTypeAnnotation;
                type = ParameterInfo.PropertyType.FILE;
                maxSize = fileAnnotation.maxSize();
                if (fileAnnotation.required()) {
                    parameterIsRequired = true;
                }
            } else if (propTypeAnnotation instanceof RequestQueryProperty) {
                RequestQueryProperty getAnnotation = (RequestQueryProperty) propTypeAnnotation;
                type = ParameterInfo.PropertyType.GET;
                if (getAnnotation.required()) {
                    parameterIsRequired = true;
                }
            } else if (propTypeAnnotation instanceof RequestPostProperty) {
                RequestPostProperty postAnnotation = (RequestPostProperty) propTypeAnnotation;
                type = ParameterInfo.PropertyType.JSON;
                if (postAnnotation.required()) {
                    parameterIsRequired = true;
                }
            } else {
                throw new RuntimeException("Unknown property type annotation " + propTypeAnnotation.getClass());
            }
            String defaultValue = null;
            if (!parameterIsRequired && instance != null) {
                try {
                    Method getter = ReflectionUtils.getGetterForName(clazz, internalName);
                    if (getter != null) {
                        Object defaultValueObj = getter.invoke(instance);
                        if (defaultValueObj != null) {
                            defaultValue = defaultValueObj.toString();
                        }
                    }
                } catch (Exception e) {
                    //ignored
                }
            }
            ParameterInfo param = new ParameterInfo(annotations, mostSpecificMethod, type, parameterName,
                    internalName, parameterIsRequired, parameterType, maxSize, defaultValue);
            result.add(param);
        }

        return result;
    }

    private static void fillSetterMethods(Class<?> clazz, Map<String, List<Method>> methods, Map<String, Field> fields) {
        Set<Class<?>> classesSet = new HashSet<>();
        Queue<Class<?>> classesQueue = new ArrayDeque<>();
        classesSet.add(clazz);
        classesQueue.offer(clazz);
        while (!classesQueue.isEmpty()) {
            Class<?> cur = classesQueue.poll();
            for (Method method : cur.getDeclaredMethods()) {
                if (ReflectionUtils.isSetter(method)) {
                    String internalName = ReflectionUtils.getNameForSetter(method);
                    methods.computeIfAbsent(internalName, ign -> new ArrayList<>())
                            .add(method);
                }
            }
            for (Field field : cur.getDeclaredFields()) {
                fields.put(field.getName(), field);
            }

            Class<?> parent = cur.getSuperclass();
            if (parent != null && classesSet.add(parent)) {
                classesQueue.add(parent);
            }

            for (Class<?> iface : cur.getInterfaces()) {
                if (classesSet.add(iface)) {
                    classesQueue.add(iface);
                }
            }
        }
    }

    private static Map<Class<? extends Annotation>, List<Annotation>> getDeclaredAnnotations(Method method) {
        Map<Class<? extends Annotation>, List<Annotation>> res = new HashMap<>();
        for (Annotation annotation : method.getDeclaredAnnotations()) {
            List<Annotation> list = res.get(annotation.annotationType());
            if (list == null) {
                list = new ArrayList<>();
                res.put(annotation.annotationType(), list);
            }
            list.add(annotation);
        }
        return res;
    }
}
