package ru.yandex.webmaster3.core.http.util;

import org.apache.commons.lang3.tuple.Pair;
import ru.yandex.webmaster3.core.http.autodoc.FullTypeInfo;

import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Queue;
import java.util.Set;
import java.util.function.Function;

/**
 * @author avhaliullin
 */
public class RequestFilterHelper<Filter> {
    private final Set<Class<?>> providedRequestTraits;
    private final List<Filter> filters;
    private final Function<Filter, FullTypeInfo> filterReqTypeProvider;
    private final List<Pair<Filter, FullTypeInfo>> filterWithRequestType;

    public RequestFilterHelper(Set<Class<?>> providedRequestTraits, List<Filter> filters,
                               Function<Filter, FullTypeInfo> filterReqTypeProvider) {
        this.providedRequestTraits = providedRequestTraits;
        this.filters = filters;
        this.filterReqTypeProvider = filterReqTypeProvider;
        this.filterWithRequestType = new ArrayList<>();
        for (Filter filter : filters) {
            filterWithRequestType.add(Pair.of(filter, filterReqTypeProvider.apply(filter)));
        }
    }

    public List<Filter> getFiltersForRequest(Class<?> requestClass) {
        List<Filter> result = new ArrayList<Filter>();
        for (Pair<Filter, FullTypeInfo> filterPair : filterWithRequestType) {
            if (filterPair.getRight().isAssignableFrom(requestClass)) {
                result.add(filterPair.getLeft());
            }
        }
        return result;
    }

    public void verifyFilterDepsTree() {
        Map<Filter, Set<Filter>> edges = new HashMap<>();
        Map<Filter, Set<Class>> filter2Traits = new HashMap<>();
        Map<Class, List<Filter>> trait2Dependencies = new HashMap<>();
        for (Filter filter : filters) {
            Set<Class> traits = new HashSet<>();
            filter2Traits.put(filter, traits);
            FullTypeInfo reqType = filterReqTypeProvider.apply(filter);
            if (reqType.isConcrete()) {
                traits.add(reqType.getClazz());
            } else {
                for (FullTypeInfo upperBound : reqType.getUpperBounds()) {
                    if (upperBound.isConcrete() && providedRequestTraits.contains(upperBound.getClazz())) {
                        traits.add(upperBound.getClazz());
                    }
                }
            }
        }

        for (Map.Entry<Filter, Set<Class>> filterWithTraits : filter2Traits.entrySet()) {
            for (Class trait : filterWithTraits.getValue()) {
                trait2Dependencies.computeIfAbsent(trait, ign -> new ArrayList<>()).add(filterWithTraits.getKey());
            }
        }

        for (Map.Entry<Filter, Set<Class>> filterWithTraits : filter2Traits.entrySet()) {
            for (Class trait : filterWithTraits.getValue()) {
                Queue<Class> ifacesQueue = new ArrayDeque<>();
                // Для самого trait'а не проверяем, кто еще на него матчится - чтобы не получить циклические зависимости на соседей и себя
                ifacesQueue.add(trait);
                while (!ifacesQueue.isEmpty()) {
                    Class cur = ifacesQueue.poll();
                    for (Class curParent : cur.getInterfaces()) {
                        if (trait2Dependencies.containsKey(curParent)) {
                            for (Filter dep : trait2Dependencies.get(curParent)) {
                                edges.computeIfAbsent(dep, ign -> new HashSet<>())
                                        .add(filterWithTraits.getKey());
                            }
                        }
                        ifacesQueue.add(curParent);
                    }
                }
            }
        }
        Pair<Filter, Filter> brokenSortEvidence =
                TopologicalSort.verifySorted(filters, edges);
        if (brokenSortEvidence != null) {
            String src = brokenSortEvidence.getLeft().getClass().getName();
            String dst = brokenSortEvidence.getRight().getClass().getName();
            throw new RuntimeException("Bad request filters order: filter " + dst + " depends on " + src + " and should follow it in filters list");
        }
    }
}
