package ru.yandex.webmaster3.core.logbroker.reader;

import java.util.List;
import java.util.concurrent.TimeUnit;

import lombok.extern.slf4j.Slf4j;

import ru.yandex.kikimr.persqueue.LogbrokerClientFactory;
import ru.yandex.kikimr.persqueue.auth.Credentials;
import ru.yandex.kikimr.persqueue.consumer.StreamConsumer;
import ru.yandex.kikimr.persqueue.consumer.internal.read.ReaderConfig;
import ru.yandex.kikimr.persqueue.consumer.stream.StreamConsumerConfig;
import ru.yandex.kikimr.persqueue.proxy.ProxyBalancer;
import ru.yandex.webmaster3.core.security.tvm.TVMTokenService;

/**
 * author: ishalaru
 * DATE: 30.07.2019
 */
@Slf4j
public class LogbrokerStreamConsumer {
    private final TVMTokenService tvmTokenService;
    private final String host;
    private final String topic;
    private final String consumePath;
    private final IDataProcessing dataProcessing;
    private ReaderConfig.Builder readerConfigBuilder;
    private StreamConsumer streamConsumer;
    private LogbrokerStreamConsumerListener consumerListener;
    private ProxyBalancer proxyBalancer;

    public LogbrokerStreamConsumer(TVMTokenService tvmTokenService,
                                   String host,
                                   String topic,
                                   String consumePath,
                                   IDataProcessing dataProcessing) {
        this.tvmTokenService = tvmTokenService;
        this.host = host;
        this.topic = topic;
        this.consumePath = consumePath;
        this.dataProcessing = dataProcessing;
    }

    public void init() {
        log.info("Logbroker stream consumer init. Config:\nhost: {}\ntopic: {}\nconsumerPath: {}", host, topic, consumePath);
        proxyBalancer = new ProxyBalancer(host);
        final LogbrokerClientFactory factory =
                new LogbrokerClientFactory(proxyBalancer);
        final StreamConsumerConfig.Builder builder = StreamConsumerConfig
                .builder(List.of(topic), consumePath)
                .setCredentialsProvider(() -> Credentials.tvm(tvmTokenService.getToken()));
        if (readerConfigBuilder != null) {
            final ReaderConfig build = readerConfigBuilder.build();
            builder.configureReader(builder1 ->
                    builder1.setMaxCount(build.getMaxCount())
                            .setMaxInflightReads(build.getMaxInflightReads())
                            .setMaxSize(build.getMaxSize())
                            .setMaxUnconsumedReads(build.getMaxUnconsumedReads())
                            .setMaxTimeLagMs(build.getMaxTimeLagMs())
                            .setPartitionsAtOnce(build.getPartitionsAtOnce())
                            .setReadDataOnlyAfterTimestampMs(build.getReadDataOnlyAfterTimestampMs())
                            .build());
        } else {
            builder.configureReader(builder1 -> builder1.setMaxUnconsumedReads(200).build());
        }

        try {
            consumerListener = new LogbrokerStreamConsumerListener(dataProcessing);
            streamConsumer = factory.streamConsumer(builder.build());
            streamConsumer.startConsume(consumerListener);
        } catch (InterruptedException e) {
            log.error("Can't create connection to logbroker.", e);
            Thread.currentThread().interrupt();
        }
    }

    public void close() {
        log.info("Logbroker stream consumer closing. Config:\nhost: {}\ntopic: {}\nconsumerPath: {}", host, topic, consumePath);
        if (streamConsumer != null) {
            try {
                streamConsumer.stopConsume();
            } catch (Exception exp) {
                log.error("{}", exp.getMessage(), exp);
            }
            try {
                proxyBalancer.shutdown(2, TimeUnit.SECONDS);
            } catch (Exception e) {
                log.error("{}", e.getMessage(), e);
            }

        }
    }

    public void checkStateAndReInit() {
        if (consumerListener == null || !consumerListener.isActive()) {
            log.info("ReInit consumer listener with parameters: {}", consumerListener);
            try {
                close();
            } catch (Exception exp) {
                log.error("{}", exp.getMessage(), exp);
            }
            init();
        }
    }

    @Override
    public String toString() {
        return "LogbrokerStreamConsumer{" +
                "host='" + host + '\'' +
                ", topic='" + topic + '\'' +
                ", consumePath='" + consumePath + '\'' +
                ", readerConfigBuilder=" + readerConfigBuilder +
                '}';
    }

    public void setReaderConfigBuilder(ReaderConfig.Builder readerConfigBuilder) {
        this.readerConfigBuilder = readerConfigBuilder;
    }
}
