package ru.yandex.webmaster3.core.metrics.externals;

import org.joda.time.Duration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.util.functional.ThrowingRunnable;
import ru.yandex.webmaster3.core.util.functional.ThrowingSupplier;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;

/**
 * @author avhaliullin
 */
public final class ExternalAPICallTracker {
    private static final Logger log = LoggerFactory.getLogger(ExternalAPICallTracker.class);

    // Можно было сразу делать IdentityHashSet, но я решил дать шанс имплементациям consumer'а
    private static final HashSet<Consumer<Event>> SUBSCRIBERS_SET = new HashSet<>();
    private static volatile List<Consumer<Event>> SUBSCRIBERS = new ArrayList<>();

    public static <R, E extends Exception> R trackQuery(String serviceName, String methodName,
                                                        Predicate<Exception> userExceptionPredicate,
                                                        ThrowingSupplier<R, E> call) throws E {

        return trackQuery(serviceName, methodName, (Exception e) -> {
            if (userExceptionPredicate.test(e)) {
                return Status.USER_ERROR;
            } else {
                return Status.INTERNAL_ERROR;
            }

        }, call);
    }

    public static <R, E extends Exception> R trackQuery(String serviceName, String methodName,
                                                        Function<Exception, Status> exceptionMappingFunction,
                                                        ThrowingSupplier<R, E> call) throws E {
        Status status = Status.INTERNAL_ERROR;
        long startNano = System.nanoTime();
        try {
            R result = call.get();
            status = Status.SUCCESS;
            return result;
        } catch (Exception e) {
            status = exceptionMappingFunction.apply(e);
            throw e;
        } finally {
            Duration duration = Duration.millis((System.nanoTime() - startNano) / 1_000_000L);
            notifySubscribers(new Event(serviceName, methodName, duration, status));
        }
    }

    public static <E extends Exception> void trackExecution(String serviceName, String methodName,
                                                            Predicate<Exception> userExceptionPredicate,
                                                            ThrowingRunnable<E> call) throws E {
        ExternalAPICallTracker.<Void, E>trackQuery(serviceName, methodName, userExceptionPredicate, () -> {
            call.run();
            return null;
        });
    }

    public static <E extends Exception> void trackExecution(String serviceName, String methodName,
                                                            Function<Exception, Status> exceptionMappingFunction,
                                                            ThrowingRunnable<E> call) throws E {
        ExternalAPICallTracker.<Void, E>trackQuery(serviceName, methodName, exceptionMappingFunction, () -> {
            call.run();
            return null;
        });
    }

    /**
     * Подписаться на все вызовы методов внешних API
     *
     * @param subscriber
     */
    public static synchronized void subscribe(Consumer<Event> subscriber) {
        //TODO: Неконтролируемый рост subscriber'ов?
        if (SUBSCRIBERS_SET.add(subscriber)) {
            List<Consumer<Event>> subscribers = new ArrayList<>(SUBSCRIBERS_SET);
            SUBSCRIBERS = subscribers;
        }
    }

    private static void notifySubscribers(Event event) {
        List<Consumer<Event>> subscribers = SUBSCRIBERS;
        for (Consumer<Event> consumer : subscribers) {
            try {
                consumer.accept(event);
            } catch (Exception e) {
                log.error("Failed to notify subscriber", e);
            }
        }
    }

    public enum Status {
        SUCCESS,
        USER_ERROR,
        INTERNAL_ERROR,
        // внешний сервис не ответил, сокет таймаут, и прочие ошибки свзяанные с соединением
        CONNECTION_ERROR
    }

    public static class Event {
        private final String serviceName;
        private final String methodName;
        private final Duration callDuration;
        private final Status status;

        public Event(String serviceName, String methodName, Duration callDuration, Status status) {
            this.serviceName = serviceName;
            this.methodName = methodName;
            this.callDuration = callDuration;
            this.status = status;
        }

        public String getServiceName() {
            return serviceName;
        }

        public String getMethodName() {
            return methodName;
        }

        public Duration getCallDuration() {
            return callDuration;
        }

        public Status getStatus() {
            return status;
        }
    }

    private ExternalAPICallTracker() { }
}
