package ru.yandex.webmaster3.core.notification;

import com.fasterxml.jackson.annotation.JsonIgnore;
import com.google.common.base.Strings;
import lombok.Value;
import org.joda.time.LocalDate;
import org.joda.time.format.DateTimeFormat;
import org.joda.time.format.DateTimeFormatter;

import ru.yandex.webmaster3.core.util.environment.YandexEnvironmentProvider;

/**
 * @author avhaliullin
 */
@Value
public class UTMLabels {
    private static final String MEDIUM_EMAIL = "email";
    private static final String MEDIUM_SERVICE = "service";
    private static final String MEDIUM_SUP = "sup";
    private static final String MEDIUM_BELL = "bell";
    private static DateTimeFormatter CONTENT_DATE_FORMAT = DateTimeFormat.forPattern("yyyy-MM-dd");

    String source;
    String medium;
    String campaign;
    String content;
    String term;
    String extendedInfo;

    @JsonIgnore
    public String getLabelsInQuery() {
        StringBuilder result = new StringBuilder();
        printLabel(result, "source", source);
        printLabel(result, "medium", medium);
        printLabel(result, "campaign", campaign);
        printLabel(result, "content", content);
        if (!Strings.isNullOrEmpty(term)) {
            printLabel(result, "term", term);
        }
        printExtendedInfo(result, extendedInfo);
        return result.toString();
    }

    private void printExtendedInfo(StringBuilder result, String value) {
        if (value != null) {
            if (result.length() > 0) {
                result.append("&");
            }
            result.append("_extended_info").append("=").append(value);
        }
    }

    private void printLabel(StringBuilder result, String labelName, String labelValue) {
        if (labelValue != null) {
            if (result.length() > 0) {
                result.append("&");
            }
            result.append("utm_").append(labelName).append("=").append(labelValue);
        }
    }

    public static UTMLabels createService(String campaign, LocalDate content, String extendedInfo) {
        return new UTMLabels(getUTMSource(), MEDIUM_SERVICE, campaign, CONTENT_DATE_FORMAT.print(content), null, extendedInfo);
    }

    public static UTMLabels createEmail(String campaign, LocalDate content, String extendedInfo) {
        return new UTMLabels(getUTMSource(), MEDIUM_EMAIL, campaign, CONTENT_DATE_FORMAT.print(content), null, extendedInfo);
    }

    public static UTMLabels createEmail(String campaign, LocalDate content, String extendedInfo, String term) {
        return new UTMLabels(getUTMSource(), MEDIUM_EMAIL, campaign, CONTENT_DATE_FORMAT.print(content), term, extendedInfo);
    }

    public static UTMLabels createSup(String campaign, LocalDate content, String extendedInfo) {
        return new UTMLabels(getUTMSource(), MEDIUM_SUP, campaign, CONTENT_DATE_FORMAT.print(content), null, extendedInfo);
    }

    public static UTMLabels createBell(String campaign, LocalDate content, String extendedInfo) {
        return new UTMLabels(getUTMSource(), MEDIUM_BELL, campaign, CONTENT_DATE_FORMAT.print(content), null, extendedInfo);
    }

    private static String getUTMSource() {
        switch (YandexEnvironmentProvider.getEnvironmentType()) {
            case PRODUCTION:
                return "webmaster.yandex.ru";
            default:
                return "webmaster.test.yandex.ru";
        }
    }
}
