package ru.yandex.webmaster3.core.regions;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.util.xml.StaxUtils;
import org.xml.sax.Attributes;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;
import ru.yandex.webmaster3.core.WebmasterException;
import ru.yandex.webmaster3.core.data.L10nEnum;
import ru.yandex.webmaster3.core.data.W3RegionInfo;
import ru.yandex.webmaster3.core.http.WebmasterErrorResponse;
import ru.yandex.webmaster3.core.util.CloseableConsumer;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;
import javax.xml.parsers.SAXParserFactory;
import javax.xml.stream.XMLEventWriter;
import javax.xml.stream.XMLOutputFactory;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamWriter;
import java.io.*;
import java.util.EnumMap;
import java.util.Map;
import java.util.function.Consumer;

/**
 * @author avhaliullin
 */
public class W3GeobaseRegionsSaver {
    private static final Logger log = LoggerFactory.getLogger(W3GeobaseRegionsSaver.class);

    private static final String TAG_REGION = "region";
    private static final String ATTRIBUTE_DEFAULT_NAME = "name";

    private static final String ATTRIBUTE_ID = "id";
    private static final String ATTRIBUTE_TYPE = "type";
    private static final String ATTRIBUTE_HIDE = "hide";
    private static final String ATTRIBUTE_PARENT = "parent";

    public static void readFrom(File file, Consumer<W3RegionInfo> consumer) throws IOException {
        try (InputStream in = new FileInputStream(file)) {
            SAXParserFactory saxParserFactory = SAXParserFactory.newInstance();
            saxParserFactory.setNamespaceAware(true);
            SAXParser parser = saxParserFactory.newSAXParser();
            parser.parse(new InputSource(in), new CachedRegionHandler(consumer));
        } catch (SAXException | ParserConfigurationException e) {
            throw new WebmasterException("Failed to parse geobase cache file.",
                    new WebmasterErrorResponse.GeobaseErrorResponse(W3GeobaseRegionsSaver.class, e), e);
        }
    }

    public static CloseableConsumer<W3RegionInfo> writeTo(File file) {
        try {
            final OutputStream out = new FileOutputStream(file);
            final XMLEventWriter xmlEventWriter = XMLOutputFactory.newFactory().createXMLEventWriter(
                    new BufferedOutputStream(out));
            XMLStreamWriter eventStreamWriter = StaxUtils.createEventStreamWriter(xmlEventWriter);
            eventStreamWriter.writeStartDocument();
            eventStreamWriter.writeStartElement("regions");
            return new CloseableConsumer<W3RegionInfo>() {
                @Override
                public void close() throws Exception {
                    try {
                        eventStreamWriter.writeEndElement();
                        eventStreamWriter.writeEndDocument();
                        eventStreamWriter.flush();
                        eventStreamWriter.close();
                    } finally {
                        out.close();
                    }
                }

                @Override
                public void accept(W3RegionInfo regionInfo) {
                    try {
                        eventStreamWriter.writeStartElement(TAG_REGION);
                        eventStreamWriter.writeAttribute(ATTRIBUTE_ID, String.valueOf(regionInfo.getId()));
                        eventStreamWriter.writeAttribute(ATTRIBUTE_PARENT, String.valueOf(regionInfo.getParentId()));
                        eventStreamWriter.writeAttribute(ATTRIBUTE_TYPE, String.valueOf(regionInfo.getType()));
                        eventStreamWriter.writeAttribute(ATTRIBUTE_DEFAULT_NAME, String.valueOf(regionInfo.getDefaultName()));
                        if (regionInfo.isHidden()) {
                            eventStreamWriter.writeAttribute(ATTRIBUTE_HIDE, "true");
                        }

                        for (Map.Entry<L10nEnum, String> entry : regionInfo.getNames().entrySet()) {
                            eventStreamWriter.writeAttribute(entry.getKey().getName(), entry.getValue());
                        }
                        eventStreamWriter.writeEndElement();
                    } catch (XMLStreamException e) {
                        throw new WebmasterException("Failed to write to geobase cache file.",
                                new WebmasterErrorResponse.GeobaseErrorResponse(W3GeobaseRegionsSaver.class, e), e);
                    }
                }
            };
        } catch (FileNotFoundException e) {
            throw new WebmasterException("Failed to open geobase cache file for write.",
                    new WebmasterErrorResponse.GeobaseErrorResponse(W3GeobaseRegionsSaver.class, e), e);
        } catch (XMLStreamException e) {
            throw new WebmasterException("Failed to write to geobase cache file.",
                    new WebmasterErrorResponse.GeobaseErrorResponse(W3GeobaseRegionsSaver.class, e), e);
        }
    }

    private static class CachedRegionHandler extends DefaultHandler {
        private final Consumer<W3RegionInfo> handler;

        public CachedRegionHandler(Consumer<W3RegionInfo> handler) {
            this.handler = handler;
        }

        private Integer getInt(Attributes attributes, String name) {
            String resString = attributes.getValue(name);
            if (resString == null || resString.isEmpty()) {
                return null;
            }
            return Integer.parseInt(resString);
        }

        @Override
        public void startElement(String uri, String localName, String qName, Attributes attributes) throws SAXException {
            if (!TAG_REGION.equalsIgnoreCase(localName)) {
                return;
            }

            Integer id = getInt(attributes, ATTRIBUTE_ID);
            if (id == null) {
                log.error("Region has no id!");
                return;
            }

            Integer type = getInt(attributes, ATTRIBUTE_TYPE);
            if (type == null) {
                log.error("Region {} has no type!", id);
                return;
            }

            Integer parentId = getInt(attributes, ATTRIBUTE_PARENT);

            EnumMap<L10nEnum, String> names = new EnumMap<>(L10nEnum.class);
            for (L10nEnum l10n : L10nEnum.values()) {
                String localizedRegionName = attributes.getValue(l10n.getName());
                if (localizedRegionName != null) {
                    names.put(l10n, localizedRegionName);
                }
            }

            String defaultName = attributes.getValue(ATTRIBUTE_DEFAULT_NAME);

            boolean hidden = null != attributes.getValue(ATTRIBUTE_HIDE);

            if (id != RegionUtils.WHOLE_WORLD_REGION.getId()) {
                handler.accept(new W3RegionInfo(parentId, id, type, names, defaultName, hidden));
            }
        }
    }
}
