package ru.yandex.webmaster3.core.searchquery;

import java.util.Optional;
import java.util.UUID;

import lombok.Getter;

import ru.yandex.webmaster3.core.data.WebmasterHostId;

/**
 * @author aherman
 */

@Getter
public class QueryGroupId {
    private final WebmasterHostId hostId;
    private final UUID groupId;
    private final SpecialGroup specialGroup;
    private final String thematics; // radar thematics

    private QueryGroupId(WebmasterHostId hostId, UUID groupId, SpecialGroup specialGroup, String thematics) {
        this.hostId = hostId;
        this.groupId = groupId;
        this.specialGroup = specialGroup;
        this.thematics = thematics;
    }

    public QueryGroupId(WebmasterHostId hostId, SpecialGroup specialGroup) {
        this(hostId, specialGroup.getGroupId(), specialGroup, "");
    }

    public QueryGroupId(WebmasterHostId hostId, UUID groupId) {
        this(hostId, groupId, SpecialGroup.UNKNOWN, "");
    }

    public static QueryGroupId createSpecialId(WebmasterHostId hostId, SpecialGroup specialGroup) {
        return new QueryGroupId(hostId, specialGroup.getGroupId(), specialGroup, "");
    }

    public static QueryGroupId createGroupId(WebmasterHostId hostId, UUID groupId) {
        return new QueryGroupId(hostId, groupId, SpecialGroup.UNKNOWN, "");
    }

    public static QueryGroupId createThematicsId(WebmasterHostId hostId, String thematics) {
        return new QueryGroupId(hostId, SpecialGroup.RIVALS_STATS.getGroupId(), SpecialGroup.RIVALS_STATS, thematics);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o) {
            return true;
        }
        if (o == null || getClass() != o.getClass()) {
            return false;
        }

        QueryGroupId groupId1 = (QueryGroupId) o;

        if (!hostId.equals(groupId1.hostId)) {
            return false;
        }
        return groupId.equals(groupId1.groupId);

    }

    @Override
    public int hashCode() {
        int result = hostId.hashCode();
        result = 31 * result + groupId.hashCode();
        return result;
    }

    @Override
    public String toString() {
        return "Group["
                + hostId.toStringId()
                + "/"
                + (isSpecial() ? specialGroup : groupId.toString())
                + "]";
    }

    public boolean isSpecial() {
        return specialGroup != SpecialGroup.UNKNOWN;
    }

    public static QueryGroupId byGroupIdStr(WebmasterHostId hostId, UUID uuid) {
        for (SpecialGroup sg : SpecialGroup.values()) {
            if (sg.getGroupId().equals(uuid)) {
                return createSpecialId(hostId, sg);
            }
        }
        return createGroupId(hostId, uuid);
    }

    public static Optional<QueryGroupId> byGroupIdStr(WebmasterHostId hostId, String groupId) {
        SpecialGroup specialGroup = SpecialGroup.R.valueOfOrUnknown(groupId);
        if (specialGroup != SpecialGroup.UNKNOWN) {
            return Optional.of(createSpecialId(hostId, specialGroup));
        }

        try {
            UUID uuid = UUID.fromString(groupId);
            return Optional.of(byGroupIdStr(hostId, uuid));
        } catch (Exception e) {
            // thematics
            return Optional.of(createThematicsId(hostId, groupId));
        }
    }
}
