package ru.yandex.webmaster3.core.searchquery;

import org.apache.commons.lang3.ArrayUtils;
import ru.yandex.webmaster3.core.util.enums.IntEnum;
import ru.yandex.webmaster3.core.util.enums.IntEnumResolver;

import java.util.EnumSet;
import java.util.Set;
import java.util.function.Function;

import static ru.yandex.webmaster3.core.searchquery.IndicatorUsage.IN_ORDER;
import static ru.yandex.webmaster3.core.searchquery.IndicatorUsage.IN_QUERY;
import static ru.yandex.webmaster3.core.searchquery.IndicatorUsage.IN_STATISTICS;

/**
 * @author aherman
 */
public enum QueryIndicator implements IntEnum {
    UNKNOWN(                        0, false),

    QUERY(                          1, false, IN_QUERY),
    URL(                            2, false, IN_QUERY),

    // Group or query name
    NAME(                           3, false, IN_ORDER),

    TOTAL_SHOWS_COUNT(              10, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    TOTAL_CLICKS_COUNT(             11, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    TOTAL_CTR(                      12, false, QueryUtils.FORMAT_PERCENT, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_SHOW_POSITION(          13, true, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_CLICK_POSITION(         14, true, IN_QUERY, IN_ORDER, IN_STATISTICS),
    TOTAL_SERPS_COUNT(              15, false),

    CLICKS_DYNAMICS(                30, TOTAL_CLICKS_COUNT, false, IN_ORDER),

    SHOWS_COUNT_1(                  80, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CLICKS_COUNT_1(                 81, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CTR_1(                          82, false, QueryUtils.FORMAT_PERCENT, IN_QUERY, IN_ORDER, IN_STATISTICS),

    SHOWS_COUNT_2_3(                90, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CLICKS_COUNT_2_3(               91, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CTR_2_3(                        92, false, QueryUtils.FORMAT_PERCENT, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_SHOW_POSITION_2_3(      93, true, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_CLICK_POSITION_2_3(     94, true, IN_QUERY, IN_ORDER, IN_STATISTICS),

    SHOWS_COUNT_4_10(               100, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CLICKS_COUNT_4_10(              101, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CTR_4_10(                       102, false, QueryUtils.FORMAT_PERCENT, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_SHOW_POSITION_4_10(     103, true, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_CLICK_POSITION_4_10(    104, true, IN_QUERY, IN_ORDER, IN_STATISTICS),

    SHOWS_COUNT_11_50(              110, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CLICKS_COUNT_11_50(             111, false, IN_QUERY, IN_ORDER, IN_STATISTICS),
    CTR_11_50(                      112, false, QueryUtils.FORMAT_PERCENT, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_SHOW_POSITION_11_50(    113, true, IN_QUERY, IN_ORDER, IN_STATISTICS),
    AVERAGE_CLICK_POSITION_11_50(   114, true, IN_QUERY, IN_ORDER, IN_STATISTICS),

    SHOWS_1_50(                     1000, false),
    CLICKS_1_50(                    1001, false),

    WEIGHTED_CLICKS(                1100, false),
    WEIGHTED_RIVALS_CLICKS(         1101, false),
    ;

    QueryIndicator(int value, boolean position, IndicatorUsage... usage) {
        this(value, null, position, usage);
    }

    QueryIndicator(int value, QueryIndicator dynamicsOf, boolean position, IndicatorUsage... usage) {
        this(value, dynamicsOf, position, Function.identity(), usage);
    }

    QueryIndicator(int value, boolean position, Function<Double, Double> formatFunction, IndicatorUsage... usage) {
        this(value, null, position, formatFunction, usage);
    }

    QueryIndicator(int value, QueryIndicator dynamicsOf, boolean position, Function<Double, Double> formatFunction,
                   IndicatorUsage... usage) {
        this.value = value;
        this.dynamicsOf = dynamicsOf;
        this.position = position;
        this.formatFunction = formatFunction;
        if (ArrayUtils.isEmpty(usage)) {
            this.usage = EnumSet.noneOf(IndicatorUsage.class);
        } else {
            this.usage = EnumSet.of(usage[0], usage);
        }
    }

    private final QueryIndicator dynamicsOf;
    private final int value;
    private final Set<IndicatorUsage> usage;
    private final boolean position;
    private final Function<Double, Double> formatFunction;

    public boolean isUsable(IndicatorUsage usage) {
        return this.usage.contains(usage);
    }

    public boolean isPosition() {
        return position;
    }

    public boolean isDynamics() {
        return dynamicsOf != null;
    }

    public QueryIndicator getDynamicsOf() {
        return dynamicsOf;
    }

    @Override
    public int value() {
        return value;
    }

    public Double format(Double value) {
        return formatFunction.apply(value);
    }

    public static IntEnumResolver<QueryIndicator> R = IntEnumResolver.r(QueryIndicator.class);
}
