package ru.yandex.webmaster3.core.searchquery.viewer;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import com.google.common.primitives.Floats;
import org.apache.commons.lang3.StringUtils;

import ru.yandex.webmaster3.core.searchquery.QueryFilter;
import ru.yandex.webmaster3.core.searchquery.QueryIndicator;

/**
 * @author aherman
 */
public class QueryFilterConverter {
    public static List<QueryFilter> toFilter(ViewerQueryFilter... filters) {
        if (filters == null || filters.length == 0) {
            return Collections.emptyList();
        }

        List<QueryFilter> result = new ArrayList<>(filters.length);
        for (ViewerQueryFilter filter : filters) {
            QueryIndicator indicator = filter.getIndicator();
            ViewerQueryFilter.Operation operation = filter.getOperation();
            String value = StringUtils.trimToEmpty(filter.getValue());

            if (indicator == QueryIndicator.QUERY || indicator == QueryIndicator.URL) {
                if (operation != ViewerQueryFilter.Operation.TEXT_CONTAINS) {
                    continue;
                }

                value = StringUtils.trimToEmpty(value);
                if (StringUtils.isEmpty(value)) {
                    continue;
                }

                if (indicator == QueryIndicator.QUERY) {
                    value = value.toLowerCase();
                }
                result.add(new QueryFilter(indicator, QueryFilter.Operation.TEXT_CONTAINS, 0, value));
            } else {
                Float valueF = Floats.tryParse(value);
                if (valueF == null) {
                    continue;
                }
                if (valueF < 0) {
                    continue;
                }
                valueF = fromCtrToFilter(indicator, valueF);
                QueryFilter.Operation op;
                switch(operation) {
                    case LESS_THAN: op = QueryFilter.Operation.LESS_THAN; break;
                    case LESS_EQUAL: op = QueryFilter.Operation.LESS_EQUAL; break;
                    case GREATER_THAN: op = QueryFilter.Operation.GREATER_THAN; break;
                    case GREATER_EQUAL: op = QueryFilter.Operation.GREATER_EQUAL; break;
                    default:
                        continue;
                }
                result.add(new QueryFilter(indicator, op, valueF, null));
            }
        }
        return result;
    }

    public static List<ViewerQueryFilter> toViewerFilter(List<QueryFilter> filters) {
        List<ViewerQueryFilter> result = new ArrayList<>();
        for (QueryFilter filter : filters) {
            ViewerQueryFilter.Operation op = null;
            String value = null;

            if (filter.getOperation() == QueryFilter.Operation.TEXT_CONTAINS) {
                value = filter.getText();
                op = ViewerQueryFilter.Operation.TEXT_CONTAINS;
            } else if (filter.getOperation() != QueryFilter.Operation.UNKNOWN) {
                float valueF = fromFilterToCtr(filter.getIndicator(), filter.getNumberValue());

                value = Float.toString(valueF);
                switch (filter.getOperation()) {
                    case GREATER_EQUAL: op = ViewerQueryFilter.Operation.GREATER_EQUAL; break;
                    case GREATER_THAN: op = ViewerQueryFilter.Operation.GREATER_THAN; break;
                    case LESS_EQUAL: op = ViewerQueryFilter.Operation.LESS_EQUAL; break;
                    case LESS_THAN: op = ViewerQueryFilter.Operation.LESS_THAN; break;
                }
            }

            if (op != null) {
                result.add(new ViewerQueryFilter(filter.getIndicator(), op, value));
            }
        }
        return result;
    }

    private static Float fromCtrToFilter(QueryIndicator indicator, float value) {
        switch (indicator) {
            case TOTAL_CTR:
            case CTR_1:
            case CTR_2_3:
            case CTR_4_10:
            case CTR_11_50:
                // TODO: Hotfix for WMC-1735
                value = value / 100;
                break;
        }
        return value;
    }

    private static float fromFilterToCtr(QueryIndicator indicator, float value) {
        switch (indicator) {
            case TOTAL_CTR:
            case CTR_1:
            case CTR_2_3:
            case CTR_4_10:
            case CTR_11_50:
                // TODO: Hotfix for WMC-1735
                value = value * 100;
                break;
        }
        return value;
    }
}
