package ru.yandex.webmaster3.core.semantic.review_business.auto.model;

import lombok.extern.slf4j.Slf4j;
import ru.yandex.webmaster3.core.semantic.review_business.ModelUtils;
import ru.yandex.webmaster3.core.semantic.review_business.model.ReviewModeration;

import java.util.Map;

import static com.google.common.base.Strings.isNullOrEmpty;
import static com.google.common.collect.Maps.newHashMap;

/**
 * Extension of ReviewModeration for reviews on autos
 * User: Evgeny Zhoga (ezhoga@yandex-team.ru)
 * Date: 4/10/12
 * Time: 5:00 PM
 */
@Slf4j
public abstract class AutoModeration extends ReviewModeration {

    /**
     * Set of rejection reasons for auto reviews
     * User: Evgeny Zhoga (ezhoga@yandex-team.ru)
     * Date: 28/9/12
     * Time: 3:31 PM
     */
    public static enum RejReason {
        AUTO_NO_EXPERIENCE("Отзыв отклонён из-за отсутствия развёрнутого описания опыта эксплуатации автомобиля."),
        AUTO_FULLNESS("Отзыв отклонён. Для публикации опишите свой опыт эксплуатации автомобиля более развёрнуто."),
        AUTO_DUMMY_EXPERIENCE("Отзыв отклонён из-за отсутствия описания личного опыта, перечислены стандартные характеристики автомобиля."),
        AUTO_PR("Отзыв отклонён из-за явного рекламного характера описания автомобиля."),
        AUTO_REVIEW_EMPTY("Отзыв отклонён. Для публикации отзыва на данную модель её описанию должна быть посвящена большая часть отзыва."),
        REVIEW_CAPS_LOCK("Отзыв отклонён из-за чрезмерного употребления заглавных букв."),
        REVIEW_EXTERNAL_LINK("Отзыв отклонён из-за наличия в тексте ссылок на сторонние ресурсы."),
        REVIEW_FOUL("Отзыв отклонён. Смягчите формулировки для публикации."),
        REVIEW_COMIC("Отзыв отклонён, так как носит шуточный характер."),
        AUTO_SALE("Отзыв отклонён. Не допускается публикация информации о покупке или продажи автомобилей."),
        REVIEW_OTHER("Отзыв отклонён из-за несоответствия требованиям размещения."),
        CUSTOM("");

        private final String text;

        private static Map<String, RejReason> message2RejReason;

        static {
            message2RejReason = newHashMap();
            for (RejReason value : values()) {
                message2RejReason.put(value.getText(), value);
            }
        }

        private RejReason(String text) {
            this.text = text;
        }

        public static RejReason safeValueOf(final String name) {
            if (isNullOrEmpty(name))
                return null;
            try {
                return RejReason.valueOf(name);
            } catch (IllegalArgumentException e) {
                log.error("Unknown rejection reason: " + name);
                return null;
            }
        }

        public static RejReason safeValueByMessage(final String name) {
            return message2RejReason.get(name);
        }

        public String getText() {
            return text;
        }
    }

    public abstract RejReason getRejReason();

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (!(o instanceof AutoModeration)) return false;
        if (!super.equals(o)) return false;

        final AutoModeration review = (AutoModeration) o;


        return hashCode() == review.hashCode()
                && ModelUtils.equals(getRejReason(), review.getRejReason())
                ;
    }

    @Override
    public int hashCode() {
        return ModelUtils.hashCodeWithNestedValue(
                super.hashCode()
                , getRejReason()
        );
    }
}

