package ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.json;

import com.google.common.base.Function;
import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.json.JSONException;
import org.json.JSONObject;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.AutoModeration;
import ru.yandex.webmaster3.core.semantic.review_business.model.impl.ReviewModerationAttributes;
import ru.yandex.webmaster3.core.semantic.review_business.util.DateHelper;


/**
 * Helps to convert {@see AutoModeration} to string serialization of JSON object and vise-versa
 *
 * @author Evgeny Zhoga <ezhoga@yandex-team.ru>
 */
@Slf4j
public final class AutoModerationJsonConversions {

    static final Function<AutoModeration, JSONObject> MODERATION_TO_JSON =
            new Function<AutoModeration, JSONObject>() {
                @Override
                public JSONObject apply(final AutoModeration arg) {
                    return toJson(arg);
                }
            };
    static final Function<JSONObject, AutoModeration> JSON_TO_MODERATION =
            new Function<JSONObject, AutoModeration>() {
                @Override
                public AutoModeration apply(final JSONObject arg) {
                    return fromJson(arg);
                }
            };

    private AutoModerationJsonConversions() {
    }

    private static AutoModeration fromJson(final JSONObject json) {
        return new AutoModerationOverJson(json);
    }

    public static AutoModerationOverJson fromJson(final String json) throws JSONException {
        return new AutoModerationOverJson(json);
    }

    public static AutoModeration safeFromJson(final String json) {
        try {
            return fromJson(json);
        } catch (JSONException e) {
            log.error("Error while parsing AutoModeration from json", e);
            return null;
        }
    }

    private static JSONObject toJson(final AutoModeration autoModeration) {
        Preconditions.checkNotNull(autoModeration, "Unable to write null AutoModeration to JSON");
        try {
            return new JSONObject().
                    putOpt(ReviewModerationAttributes.STATUS, autoModeration.getStatus()).
                    putOpt(ReviewModerationAttributes.REJECTION_REASON, autoModeration.getRejReason()).
                    putOpt(ReviewModerationAttributes.CUSTOM_REJECTION_TEXT, autoModeration.getCustomRejectionText()).
                    putOpt(ReviewModerationAttributes.HANDLER, autoModeration.getHandler()).
                    putOpt(ReviewModerationAttributes.TIME, DateHelper.safeWrite(autoModeration.getTime())).
                    putOpt(ReviewModerationAttributes.NOTE, autoModeration.getNote());
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }
    }

    public static String toJsonString(final AutoModeration autoModeration) {
        return toJson(autoModeration).toString();
    }
}

