package ru.yandex.webmaster3.core.semantic.review_business.auto.model.ugc;

import ru.yandex.webmaster3.core.semantic.review_business.ModelUtils;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.BaseAutoReview;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.ugc.impl.json.UgcAutoReviewJsonConversions;
import ru.yandex.webmaster3.core.semantic.review_business.model.UserIdentity;

import java.util.Locale;

/**
 * Represents UGC Auto review
 * User: Evgeny Zhoga (ezhoga@yandex-team.ru)
 * Date: 15/10/12
 * Time: 6:45 PM
 */
public abstract class UgcAutoReview extends BaseAutoReview {
    /**
     * UID is primary key in the passport database.
     * For details see
     * http://doc.yandex-team.ru/Passport/AuthDevGuide/concepts/Account_About.xml
     *
     * @return review author's UID
     */
    public abstract Long getUid();

    /**
     * @return String representation of Owning time (like "1-2 years")
     */
    public abstract String getOwningTime();

    /**
     * @return review created date in milliseconds
     */
    public abstract Long getCreatedTime();

    /**
     * This method will be removed lately after migrating UGC to mysql
     *
     * @return review last modification date in milliseconds
     */
    public abstract Long getLastModificationTime();

    /**
     * Information about user environment at the time of review creation:
     * * User-Agent
     * * Cookies
     * * IP
     * * etc...
     */
    public abstract UserIdentity getUserIdentity();

    /**
     * ID of previous version of review from a user on an auto or
     * null if this is a first version.
     */
    public abstract String getPreviousId();

    /**
     * User domain (from which domain review has been written).
     */
    public abstract String getDomain();

    /**
     * User interface language (which language was selected at preferences at the moment of review creation)
     */
    public abstract Locale getUiLanguage();

    @Override
    public boolean equals(final Object o) {
        if (this == o) return true;
        if (!(o instanceof UgcAutoReview)) return false;
        if (!super.equals(o)) return false;

        final UgcAutoReview review = (UgcAutoReview) o;

        return hashCode() == review.hashCode()
                && ModelUtils.equals(getUid(), review.getUid())
                && ModelUtils.equals(getOwningTime(), review.getOwningTime())
                && ModelUtils.equals(getCreatedTime(), review.getCreatedTime())
                && ModelUtils.equals(getLastModificationTime(), review.getLastModificationTime())
                && ModelUtils.equals(getUserIdentity(), review.getUserIdentity())
                && ModelUtils.equals(getPreviousId(), review.getPreviousId())
                && ModelUtils.equals(getPhotoUrls(), review.getPhotoUrls())
                && ModelUtils.equals(getDomain(), review.getDomain())
                && ModelUtils.equals(getUiLanguage(), review.getUiLanguage());
    }

    @Override
    public int hashCode() {
        return ModelUtils.hashCodeWithNestedValue(
                super.hashCode()
                , getUid()
                , getOwningTime()
                , getCreatedTime()
                , getLastModificationTime()
                , getUserIdentity()
                , getPreviousId()
                , getPhotoUrls()
                , getDomain()
                , getUiLanguage()
        );
    }

    /**
     * Construct string representation of UgcAutoReview.
     */
    @Override
    public String toString() {
        return UgcAutoReviewJsonConversions.toJsonString(this);
    }
}
