package ru.yandex.webmaster3.core.semantic.review_business.auto.model.ugc.impl.json;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.json.JSONException;
import org.json.JSONObject;
import ru.yandex.webmaster3.core.semantic.review_business.Locales;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.*;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.BaseAutoReviewAttributes;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.impl.json.BaseAutoReviewOverJson;
import ru.yandex.webmaster3.core.semantic.review_business.auto.model.ugc.UgcAutoReview;
import ru.yandex.webmaster3.core.semantic.review_business.model.Author;
import ru.yandex.webmaster3.core.semantic.review_business.model.UserIdentity;
import ru.yandex.webmaster3.core.semantic.review_business.model.impl.json.UserIdentityOverJson;
import ru.yandex.webmaster3.core.semantic.review_business.util.JsonAccessUtils;

import java.util.*;

import static com.google.common.base.Strings.emptyToNull;
import static ru.yandex.webmaster3.core.semantic.review_business.auto.model.ugc.impl.UgcAutoReviewAttributes.*;

/**
 * User: Evgeny Zhoga (ezhoga@yandex-team.ru)
 * Date: 15/10/12
 * Time: 7:07 PM
 */
@Slf4j
public class UgcAutoReviewOverJson extends UgcAutoReview {
    private final JSONObject json;
    private final BaseAutoReview review;
    private final UserIdentity userIdentity;

    public UgcAutoReviewOverJson(String json) throws JSONException {
        Preconditions.checkNotNull(emptyToNull(json), "Unable to create UgcAutoReview over empty JSON string");

            this.json = new JSONObject(json);
            review = new BaseAutoReviewOverJsonWithSharedDelegate(this.json);
            this.userIdentity = this.json.has(USER_IDENTITY) ?
                    new UserIdentityOverJson(this.json.optJSONObject(USER_IDENTITY)) : null;
    }

    public UgcAutoReviewOverJson(JSONObject json){
        Preconditions.checkNotNull(json, "Unable to create UgcAutoReview over null JSON");

        review = new BaseAutoReviewOverJsonWithSharedDelegate(json);
        this.json = json;
        this.userIdentity = json.has(USER_IDENTITY) ?
                new UserIdentityOverJson(json.optJSONObject(USER_IDENTITY)) : null;
    }

    @Override
    public String getLanguage() {
        return review.getLanguage();
    }

    @Override
    public String getId() {
        return review.getId();
    }

    @Override
    public String getPermanentId() {
        return review.getPermanentId();
    }

    @Override
    public String getVersionId() {
        return review.getVersionId();
    }

    @Override
    public Float getRating() {
        return review.getRating();
    }

    @Override
    public Float getBestRating() {
        return review.getBestRating();
    }

    @Override
    public Float getWorstRating() {
        return review.getWorstRating();
    }

    @Override
    public Float getNormalizedRating() {
        return review.getNormalizedRating();
    }

    @Override
    public Author getReviewer() {
        return review.getReviewer();
    }

    @Override
    public Date getReviewedDate() {
        return review.getReviewedDate();
    }

    @Override
    public String getDescription() {
        return review.getDescription();
    }

    @Override
    public String getSummary() {
        return review.getSummary();
    }

    @Override
    public Date getWatchTime() {
        return review.getWatchTime();
    }

    @Override
    public HProduct getItem() {
        return review.getItem();
    }

    @Override
    public Collection<String> getPhotos() {
        return review.getPhotos();
    }

    @Override
    public Collection<String> getPros() {
        return review.getPros();
    }

    @Override
    public Collection<String> getContras() {
        return review.getContras();
    }

    @Override
    public List<AutoModeration> getModerationHistory() {
        return review.getModerationHistory();
    }

    @Override
    public Long getUid() {
        return JsonAccessUtils.optLong(json, YUID);
    }

    @Override
    public boolean isDeleted() {
        return review.isDeleted();
    }

    @Override
    public String getOwningTime() {
        return json.optString(OWNING_TIME, null);
    }

    @Override
    public Long getCreatedTime() {
        return JsonAccessUtils.optLong(json, CREATED_TIME);
    }

    @Override
    public Long getLastModificationTime() {
        return JsonAccessUtils.optLong(json, LAST_MODIFICATION_TIME);
    }

    @Override
    public UserIdentity getUserIdentity() {
        return userIdentity;
    }

    @Override
    public String getPreviousId() {
        return json.optString(PREVIOUS_ID, null);
    }

    @Override
    public String getDomain() {
        return json.optString(DOMAIN, null);
    }

    @Override
    public Locale getUiLanguage() {
        return Locales.safeValueOf(json.optString(UI_LANGUAGE, null));
    }
    @Override
    public Integer getUsefulCount() {
        return review.getUsefulCount();
    }

    @Override
    public Integer getUselessCount() {
        return review.getUselessCount();
    }

    @Override
    public List<String> getPhotoUrls() {
        try {
            return JsonAccessUtils.arrayToStrings(json.optJSONArray(BaseAutoReviewAttributes.PHOTO_URLS));
        } catch (JSONException e) {
            log.error("Error while getting photo-urls", e);
            return Collections.emptyList();
        }

    }

    /**
     * Inner child of BaseAutoReviewOverJson to get access to contructor with JSONObject argument
     */
    private static final class BaseAutoReviewOverJsonWithSharedDelegate extends BaseAutoReviewOverJson {
        BaseAutoReviewOverJsonWithSharedDelegate(JSONObject json) {
            super(json);
        }
    }
}

