package ru.yandex.webmaster3.core.semantic.review_business.biz.model.impl.json;

import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.*;
import ru.yandex.webmaster3.core.semantic.review_business.model.Author;
import ru.yandex.webmaster3.core.semantic.review_business.model.LinkToOriginReview;
import ru.yandex.webmaster3.core.semantic.review_business.model.impl.json.AuthorJsonConversions;
import ru.yandex.webmaster3.core.semantic.review_business.model.impl.json.LinkToOriginReviewJsonConversions;
import ru.yandex.webmaster3.core.semantic.review_business.util.DateHelper;
import ru.yandex.webmaster3.core.semantic.review_business.util.JsonAccessUtils;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.BizReview;

import java.util.*;

import static com.google.common.base.Strings.emptyToNull;
import static ru.yandex.webmaster3.core.semantic.review_business.biz.model.impl.BizReviewAttributes.*;

/**
 * TODO
 *
 * @author Dima Schitinin <dimas@yandex-team.ru>
 */
public class BizReviewOverJson extends BizReview {

    private static final Logger log = LoggerFactory.getLogger(BizReviewOverJson.class);

    private final JSONObject json;

    /**
     * This implementation breaks the contract for hashCode (because hashCode is counted
     * on top of all get methods of Object and JSONObject is mutable and cannot be controlled from
     * this implementation)
     * Visibility will be changed to package private (in case of necessity
     * of usage in constructing of high-level objects) or removed completely
     */
    @Deprecated
    public BizReviewOverJson(final JSONObject json) {
        Preconditions.checkNotNull(json, "Unable to create BizReview over null JSON");
        this.json = json;
    }

    public BizReviewOverJson(final String json) throws JSONException {
        Preconditions.checkNotNull(emptyToNull(json), "Unable to create BizReview over empty JSON string");
        this.json = new JSONObject(json);
    }

    @Override
    public String getLanguage() {
        return json.optString(LANGUAGE, null);
    }

    @Override
    public String getId() {
        return json.optString(ID, null);
    }

    @Override
    public Long getBizId() {
        return JsonAccessUtils.optLong(json, BIZ_ID);
    }

    @Override
    public Long getAutoBizId() {
        return JsonAccessUtils.optLong(json, AUTO_BIZ_ID);
    }

    @Override
    public Set<Long> getExcludedBizIds() {
        return Sets.newHashSet(JsonAccessUtils.arrayToLongs(json.optJSONArray(EXCLUDED_BIZ_IDS)));
    }

    @Override
    public Long getChainId() {
        return JsonAccessUtils.optLong(json, CHAIN_ID);
    }

    @Override
    public BizCategory getCategory() {
        return BizCategory.byName(json.optString(CATEGORY, null), null);
    }

    @Override
    public Collection<String> getBizRubrics() {
        try {
            return JsonAccessUtils.arrayToStrings(json.optJSONArray(BIZ_RUBRICS));
        } catch (JSONException e) {
            log.error("Error while getting biz rubrics", e);
            return Collections.emptyList();
        }
    }

    @Override
    public Float getRating() {
        return JsonAccessUtils.optFloat(json, RATING);
    }

    @Override
    public Float getBestRating() {
        return JsonAccessUtils.optFloat(json, BEST_RATING);
    }

    @Override
    public Float getWorstRating() {
        return JsonAccessUtils.optFloat(json, WORST_RATING);
    }

    @Override
    public Author getReviewer() {
        final JSONObject authorAsJson = json.optJSONObject(REVIEWER);
        return authorAsJson == null ?
                null :
                AuthorJsonConversions.fromJson(authorAsJson);
    }

    @Override
    public HCard getItem() {
        final JSONObject itemAsJson = json.optJSONObject(ITEM);
        return itemAsJson == null ?
                null :
                HCardJsonConversions.fromJson(itemAsJson);
    }

    @Override
    public String getSummary() {
        return json.optString(SUMMARY, null);
    }

    @Override
    public String getDescription() {
        return json.optString(DESCRIPTION, null);
    }

    @Override
    public Collection<String> getPros() {
        try {
            return JsonAccessUtils.arrayToStrings(json.optJSONArray(PROS));
        } catch (JSONException e) {
            log.error("Error while getting pros", e);
            return Collections.emptyList();
        }
    }

    @Override
    public Collection<String> getContras() {
        try {
            return JsonAccessUtils.arrayToStrings(json.optJSONArray(CONTRAS));
        } catch (JSONException e) {
            log.error("Error while getting contras", e);
            return Collections.emptyList();
        }
    }

    @Override
    public Integer getUsefulCount() {
        return JsonAccessUtils.optInt(json, USEFUL);
    }

    @Override
    public Integer getUselessCount() {
        return JsonAccessUtils.optInt(json, USELESS);
    }

    @Override
    public Integer getLikesCount() {
        return JsonAccessUtils.optInt(json, LIKES);
    }

    @Override
    public Integer getCommentsCount() {
        return JsonAccessUtils.optInt(json, COMMENTS);
    }

    @Override
    public Date getReviewedDate() {
        return DateHelper.read(json.optString(REVIEWED_DATE));
    }

    @Override
    public String getVisitedDate() {
        return json.optString(VISITED_DATE, null);
    }

    @Override
    public String getReviewsUrl() {
        return json.optString(REVIEWS_URL, null);
    }

    @Override
    public String getCommentsUrl() {
        return json.optString(COMMENT_URL, null);
    }

    @Override
    public Collection<Tag> getTags() {
        try {
            return Collections2.transform(
                    JsonAccessUtils.arrayToJsons(json.optJSONArray(TAGS)),
                    TagJsonConversions.JSON_TO_TAG);
        } catch (JSONException e) {
            log.error("Error while getting tags from JSON", e);
            return Collections.emptyList();
        }
    }

    @Override
    public Collection<Tag> getFacts() {
        try {
            return Collections2.transform(
                    JsonAccessUtils.arrayToJsons(json.optJSONArray(FACTS)),
                    TagJsonConversions.JSON_TO_TAG);
        } catch (JSONException e) {
            log.error("Error while getting facts from JSON", e);
            return Collections.emptyList();
        }
    }

    @Override
    public String getUrl() {
        return json.optString(URL, null);
    }

    @Override
    public LinkToOriginReview getOriginReview() {
        final JSONObject originReview = json.optJSONObject(ORIGIN_REVIEW);
        if (originReview == null) {
            return null;
        } else {
            return LinkToOriginReviewJsonConversions.fromJson(originReview);
        }
    }

    @Override
    public Long getSourceId() {
        return JsonAccessUtils.optLong(json, SOURCE_ID);
    }

    @Override
    public String getPermanentId() {
        return json.optString(PERMANENT_ID, null);
    }

    @Override
    public String getVersionId() {
        return json.optString(VERSION_ID, null);
    }

    @Override
    public boolean isDeleted() {
        return json.optBoolean(IS_DELETED, false);
    }

    @Override
    public List<Moderation2> getModerationHistory() {
        try {
            return Lists.transform(
                    JsonAccessUtils.arrayToJsons(json.optJSONArray(MODERATION_HISTORY)),
                    Moderation2JsonConversions.JSON_TO_MODERATION);
        } catch (JSONException e) {
            log.error("Error while getting moderation history from JSON", e);
            return Collections.emptyList();
        }
    }

    @Override
    public Date getWatchTime() {
        return DateHelper.read(json.optString(WATCH_TIME));
    }

    @Override
    public String getSpecialistId() {
        return json.optString(SPECIALIST_ID, null);
    }

    @Override
    public boolean isVerified() {
        return json.optBoolean(IS_VERIFIED, false);
    }

    @Override
    public Geo getGeo() {
        if (json.optJSONObject(GEO) == null)
            return null;
        try {
            return new GeoOverJson(json.optJSONObject(GEO).toString());
        } catch (JSONException e) {
            return null;
        }
    }
}

