package ru.yandex.webmaster3.core.semantic.review_business.biz.model.impl.json;

import com.google.common.base.Preconditions;
import com.google.common.collect.Collections2;
import org.json.JSONArray;
import org.json.JSONException;
import org.json.JSONObject;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import ru.yandex.webmaster3.core.semantic.review_business.biz.model.HCard;
import ru.yandex.webmaster3.core.semantic.review_business.util.DateHelper;

import static org.apache.commons.collections.CollectionUtils.isEmpty;
import static ru.yandex.webmaster3.core.semantic.review_business.biz.model.impl.HCardAttributes.*;

/**
 * Helps to convert {@link HCard} to
 * {@link JSONObject} and vise versa.
 *
 * @author Dima Schitinin <dimas@yandex-team.ru>
 */
public final class HCardJsonConversions {

    private static final Logger log = LoggerFactory.getLogger(HCardJsonConversions.class);

    private HCardJsonConversions() {
    }

    /**
     * Use {@link #fromJson(String)} (throws {@link JSONException}) and {@link #safeFromJson(String)} instead
     */
    @Deprecated
    public static HCard fromJson(final JSONObject json) {
        return new HCardOverJson(json);
    }

    public static HCard fromJson(final String json) throws JSONException {
        return new HCardOverJson(json);
    }

    public static HCard safeFromJson(final String json) {
        try {
            return fromJson(json);
        } catch (JSONException e) {
            log.error("Error while parsing HCard from json", e);
            return null;
        }
    }

    public static JSONObject toJson(final HCard hCard) {
        Preconditions.checkNotNull(hCard, "Unable to convert null HCard to JSON");
        try {
            return new JSONObject().
                    putOpt(FN, hCard.getFn()).
                    putOpt(ALTNAMES, isEmpty(hCard.getAltNames()) ? null : new JSONArray(hCard.getAltNames())).
                    putOpt(ADRS, isEmpty(hCard.getAddresses()) ?
                            null : new JSONArray(Collections2.transform(hCard.getAddresses(), AddressJsonConversions.ADDRESS_TO_JSON))).
                    putOpt(URL, hCard.getUrl()).
                    putOpt(LOCALURL, hCard.getLocalUrl()).
                    putOpt(EMAIL, hCard.getEmail()).
                    putOpt(TEL, hCard.getPhoneNumber()).
                    putOpt(WORKHOURS, hCard.getWorkHours()).
                    putOpt(PHOTO, hCard.getPhotoUrl()).
                    putOpt(LOGO, hCard.getLogoUrl()).
                    putOpt(FEATURES, isEmpty(hCard.getFeatures()) ?
                            null : new JSONArray(Collections2.transform(hCard.getFeatures(), FeatureJsonConversions.FEATURE_TO_JSON))).
                    putOpt(GEO, hCard.getGeo() != null ? GeoJsonConversions.toJson(hCard.getGeo()) : null).
                    putOpt(BIZ_ID, hCard.getBizId()).
                    putOpt(AUTO_BIZ_ID, hCard.getAutoBizId()).
                    putOpt(EXCLUDED_BIZ_IDS, !hCard.getExcludedBizIds().isEmpty() ? new JSONArray(hCard.getExcludedBizIds()) : null).
                    putOpt(WATCH_TIME,
                            hCard.getWatchTime() == null ?
                                    null : DateHelper.write(hCard.getWatchTime())).
                    putOpt(SOURCE_ID, hCard.getSourceId()).
                    putOpt(CATEGORY, hCard.getCategory() != null ? hCard.getCategory().getName() : null);
        } catch (JSONException e) {
            throw new RuntimeException(e);
        }
    }

    public static String toJsonString(final HCard hCard) {
        return toJson(hCard).toString();
    }

    @Deprecated
    public static JSONObject safeToJson(final HCard hCard) {
        return toJson(hCard);
    }

    @Deprecated
    public static String safeToJsonString(final HCard hCard) {
        return toJsonString(hCard);
    }
}

